/**
 * PDF Template Designer JavaScript
 * Handles drag-and-drop template building
 */

let canvasElements = [];
let selectedElement = null;
let elementIdCounter = 0;

// Show/hide designer view
function showDesigner() {
    document.getElementById('template-list-view').style.display = 'none';
    document.getElementById('template-designer-view').style.display = 'block';
    resetDesigner();
}

function hideDesigner() {
    if (confirm('Any unsaved changes will be lost. Continue?')) {
        document.getElementById('template-designer-view').style.display = 'none';
        document.getElementById('template-list-view').style.display = 'block';
    }
}

function resetDesigner() {
    canvasElements = [];
    selectedElement = null;
    document.getElementById('canvas-area').innerHTML = `
        <div style="text-align: center; color: var(--text-muted); padding: 4rem 2rem;">
            <div style="font-size: 3rem; margin-bottom: 1rem;">📄</div>
            <p>Drag and drop fields here to build your template</p>
        </div>
    `;
    document.getElementById('template-form').reset();
    document.getElementById('template_id').value = '';
    document.getElementById('properties-content').innerHTML = '';
}

// Initialize drag and drop
document.addEventListener('DOMContentLoaded', function() {
    const canvas = document.getElementById('canvas-area');
    if (!canvas) return;
    
    // Make fields draggable
    document.querySelectorAll('.field-item').forEach(item => {
        item.addEventListener('dragstart', handleDragStart);
    });
    
    // Canvas drop handlers
    canvas.addEventListener('dragover', handleDragOver);
    canvas.addEventListener('drop', handleDrop);
    canvas.addEventListener('dragleave', handleDragLeave);
});

function handleDragStart(e) {
    e.dataTransfer.effectAllowed = 'copy';
    e.dataTransfer.setData('text/plain', JSON.stringify({
        table: this.dataset.table,
        field: this.dataset.field,
        label: this.dataset.label,
        type: this.dataset.type
    }));
}

function handleDragOver(e) {
    e.preventDefault();
    e.dataTransfer.dropEffect = 'copy';
    e.currentTarget.classList.add('dragover');
}

function handleDragLeave(e) {
    e.currentTarget.classList.remove('dragover');
}

function handleDrop(e) {
    e.preventDefault();
    e.currentTarget.classList.remove('dragover');
    
    const data = JSON.parse(e.dataTransfer.getData('text/plain'));
    addElementToCanvas(data);
}

function addElementToCanvas(fieldData) {
    const element = {
        id: 'element-' + (elementIdCounter++),
        type: 'field',
        source: fieldData.table,
        field: fieldData.field,
        label: fieldData.label,
        fieldType: fieldData.type,
        fontSize: 12,
        bold: false,
        color: '#333333',
        showLabel: true
    };
    
    canvasElements.push(element);
    renderCanvas();
}

function renderCanvas() {
    const canvas = document.getElementById('canvas-area');
    
    if (canvasElements.length === 0) {
        canvas.innerHTML = `
            <div style="text-align: center; color: var(--text-muted); padding: 4rem 2rem;">
                <div style="font-size: 3rem; margin-bottom: 1rem;">📄</div>
                <p>Drag and drop fields here to build your template</p>
            </div>
        `;
        return;
    }
    
    canvas.innerHTML = '';
    
    canvasElements.forEach((element, index) => {
        const div = document.createElement('div');
        div.className = 'canvas-element';
        div.id = element.id;
        if (selectedElement && selectedElement.id === element.id) {
            div.classList.add('selected');
        }
        
        div.innerHTML = `
            <div class="element-controls">
                <button class="element-control-btn" onclick="moveElementUp(${index})" ${index === 0 ? 'disabled' : ''}>↑</button>
                <button class="element-control-btn" onclick="moveElementDown(${index})" ${index === canvasElements.length - 1 ? 'disabled' : ''}>↓</button>
                <button class="element-control-btn" onclick="removeElement(${index})">✕</button>
            </div>
            <div style="margin-right: 60px;">
                ${element.showLabel ? `<strong>${element.label}:</strong><br>` : ''}
                <span style="font-size: ${element.fontSize}px; ${element.bold ? 'font-weight: bold;' : ''} color: ${element.color};">
                    {${element.source}.${element.field}}
                </span>
            </div>
        `;
        
        div.addEventListener('click', function() {
            selectElement(element);
        });
        
        canvas.appendChild(div);
    });
}

function selectElement(element) {
    selectedElement = element;
    renderCanvas();
    loadElementProperties(element);
}

function loadElementProperties(element) {
    const content = document.getElementById('properties-content');
    
    content.innerHTML = `
        <div class="property-group">
            <label class="property-label">Label Text</label>
            <input type="text" class="form-control" value="${element.label}" 
                   onchange="updateElementProperty('label', this.value)">
        </div>
        
        <div class="property-group">
            <div style="display: flex; align-items: center; gap: 0.5rem;">
                <input type="checkbox" ${element.showLabel ? 'checked' : ''} 
                       onchange="updateElementProperty('showLabel', this.checked)" style="width: auto;">
                <label style="margin: 0;">Show Label</label>
            </div>
        </div>
        
        <div class="property-group">
            <label class="property-label">Font Size</label>
            <input type="number" class="form-control" value="${element.fontSize}" min="8" max="72"
                   onchange="updateElementProperty('fontSize', parseInt(this.value))">
        </div>
        
        <div class="property-group">
            <div style="display: flex; align-items: center; gap: 0.5rem;">
                <input type="checkbox" ${element.bold ? 'checked' : ''} 
                       onchange="updateElementProperty('bold', this.checked)" style="width: auto;">
                <label style="margin: 0;">Bold</label>
            </div>
        </div>
        
        <div class="property-group">
            <label class="property-label">Text Color</label>
            <input type="color" class="form-control" value="${element.color}"
                   onchange="updateElementProperty('color', this.value)">
        </div>
        
        <div class="property-group">
            <label class="property-label">Field Info</label>
            <div style="background: var(--bg-tertiary); padding: 0.75rem; border-radius: 6px; font-size: 0.85rem;">
                <div><strong>Source:</strong> ${element.source}</div>
                <div><strong>Field:</strong> ${element.field}</div>
                <div><strong>Type:</strong> ${element.fieldType}</div>
            </div>
        </div>
    `;
}

function updateElementProperty(property, value) {
    if (selectedElement) {
        selectedElement[property] = value;
        renderCanvas();
        loadElementProperties(selectedElement);
    }
}

function moveElementUp(index) {
    if (index > 0) {
        [canvasElements[index], canvasElements[index - 1]] = [canvasElements[index - 1], canvasElements[index]];
        renderCanvas();
    }
}

function moveElementDown(index) {
    if (index < canvasElements.length - 1) {
        [canvasElements[index], canvasElements[index + 1]] = [canvasElements[index + 1], canvasElements[index]];
        renderCanvas();
    }
}

function removeElement(index) {
    if (confirm('Remove this element?')) {
        canvasElements.splice(index, 1);
        selectedElement = null;
        renderCanvas();
        document.getElementById('properties-content').innerHTML = '';
    }
}

function previewTemplate() {
    if (canvasElements.length === 0) {
        alert('Add some fields to the template first');
        return;
    }
    
    const config = buildTemplateConfig();
    alert('Preview functionality will open a new window with the template preview.\n\n' + 
          'Fields: ' + canvasElements.length);
}

function saveTemplate() {
    // Validate
    const name = document.getElementById('template_name').value;
    if (!name) {
        alert('Please enter a template name');
        document.getElementById('template_name').focus();
        return;
    }
    
    if (canvasElements.length === 0) {
        if (!confirm('Template has no fields. Save anyway?')) {
            return;
        }
    }
    
    // Build configuration
    const config = buildTemplateConfig();
    document.getElementById('template_config').value = JSON.stringify(config);
    
    // Submit form
    document.getElementById('template-form').submit();
}

function buildTemplateConfig() {
    return {
        header: {
            enabled: true,
            title: 'Risk Assessment Report',
            height: 60
        },
        footer: {
            enabled: true,
            center_text: 'Page {page} of {total}',
            height: 40
        },
        sections: canvasElements,
        styling: {
            margins: {
                top: 20,
                right: 20,
                bottom: 20,
                left: 20
            },
            fontFamily: 'Helvetica',
            primaryColor: '#2563eb'
        }
    };
}

function editTemplate(templateId) {
    // Load template data via AJAX and populate designer
    fetch('api/get-template.php?id=' + templateId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                populateDesigner(data.template);
                showDesigner();
            } else {
                alert('Error loading template: ' + data.error);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading template');
        });
}

function populateDesigner(template) {
    document.getElementById('template_id').value = template.id;
    document.getElementById('template_name').value = template.template_name;
    document.getElementById('description').value = template.description || '';
    document.getElementById('template_type').value = template.template_type;
    document.getElementById('page_size').value = template.page_size;
    document.getElementById('orientation').value = template.orientation;
    document.getElementById('is_default').checked = template.is_default == 1;
    document.getElementById('is_active').checked = template.is_active == 1;
    
    const config = JSON.parse(template.template_config);
    if (config.sections) {
        canvasElements = config.sections;
        elementIdCounter = canvasElements.length;
        renderCanvas();
    }
}

function duplicateTemplate(templateId) {
    if (confirm('Create a copy of this template?')) {
        fetch('api/get-template.php?id=' + templateId)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const template = data.template;
                    populateDesigner(template);
                    document.getElementById('template_id').value = ''; // Clear ID for new template
                    document.getElementById('template_name').value = template.template_name + ' (Copy)';
                    document.getElementById('is_default').checked = false;
                    showDesigner();
                }
            });
    }
}
