<?php
require_once '../includes/config.php';
requireLogin();

$editing = false;
$location = null;
$error = '';
$success = '';

// Check if editing
if (isset($_GET['id'])) {
    $editing = true;
    $stmt = $pdo->prepare("SELECT * FROM locations WHERE id = ?");
    $stmt->execute([$_GET['id']]);
    $location = $stmt->fetch();
    
    if (!$location) {
        header('Location: locations.php');
        exit;
    }
    
    $location['opening_hours'] = json_decode($location['opening_hours'], true) ?: [];
    $location['delivery_windows'] = json_decode($location['delivery_windows'], true) ?: [];
    $location['custom_fields'] = json_decode($location['custom_fields'], true) ?: [];
}

// Fetch custom field definitions
$stmt = $pdo->query("SELECT * FROM custom_field_definitions WHERE is_active = 1 ORDER BY display_order, field_label");
$custom_field_defs = $stmt->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Prepare opening hours JSON
        $opening_hours = [];
        $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        foreach ($days as $day) {
            if (!empty($_POST["hours_{$day}_open"]) || !empty($_POST["hours_{$day}_close"])) {
                $opening_hours[$day] = [
                    'open' => $_POST["hours_{$day}_open"] ?? '',
                    'close' => $_POST["hours_{$day}_close"] ?? ''
                ];
            }
        }
        
        // Prepare delivery windows JSON
        $delivery_windows = [];
        if (!empty($_POST['delivery_window_start'])) {
            foreach ($_POST['delivery_window_start'] as $idx => $start) {
                if (!empty($start) && !empty($_POST['delivery_window_end'][$idx])) {
                    $delivery_windows[] = [
                        'start' => $start,
                        'end' => $_POST['delivery_window_end'][$idx],
                        'days' => $_POST['delivery_window_days'][$idx] ?? ''
                    ];
                }
            }
        }
        
        // Prepare custom fields JSON
        $custom_fields = [];
        if (!empty($_POST['custom_field'])) {
            foreach ($_POST['custom_field'] as $field_name => $value) {
                if ($value !== '' && $value !== null) {
                    $custom_fields[$field_name] = $value;
                }
            }
        }
        
        if ($editing) {
            // Update existing location
            $stmt = $pdo->prepare("
                UPDATE locations SET 
                    location_name = ?, address_line1 = ?, address_line2 = ?, city = ?, postcode = ?,
                    region = ?, contact_name = ?, contact_phone = ?, contact_email = ?, notes = ?,
                    opening_hours = ?, delivery_windows = ?, latitude = ?, longitude = ?,
                    what3words = ?, google_maps_place_id = ?, google_maps_url = ?, 
                    google_street_view_url = ?, custom_fields = ?
                WHERE id = ?
            ");
            
            $stmt->execute([
                $_POST['location_name'],
                $_POST['address_line1'],
                $_POST['address_line2'] ?? '',
                $_POST['city'],
                $_POST['postcode'],
                $_POST['region'] ?? '',
                $_POST['contact_name'] ?? '',
                $_POST['contact_phone'] ?? '',
                $_POST['contact_email'] ?? '',
                $_POST['notes'] ?? '',
                json_encode($opening_hours),
                json_encode($delivery_windows),
                !empty($_POST['latitude']) ? $_POST['latitude'] : null,
                !empty($_POST['longitude']) ? $_POST['longitude'] : null,
                $_POST['what3words'] ?? '',
                $_POST['google_maps_place_id'] ?? '',
                $_POST['google_maps_url'] ?? '',
                $_POST['google_street_view_url'] ?? '',
                json_encode($custom_fields),
                $_GET['id']
            ]);
            
            $success = 'Location updated successfully!';
        } else {
            // Insert new location
            $stmt = $pdo->prepare("
                INSERT INTO locations (location_name, address_line1, address_line2, city, postcode,
                                     region, contact_name, contact_phone, contact_email, notes,
                                     opening_hours, delivery_windows, latitude, longitude,
                                     what3words, google_maps_place_id, google_maps_url, 
                                     google_street_view_url, custom_fields)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $_POST['location_name'],
                $_POST['address_line1'],
                $_POST['address_line2'] ?? '',
                $_POST['city'],
                $_POST['postcode'],
                $_POST['region'] ?? '',
                $_POST['contact_name'] ?? '',
                $_POST['contact_phone'] ?? '',
                $_POST['contact_email'] ?? '',
                $_POST['notes'] ?? '',
                json_encode($opening_hours),
                json_encode($delivery_windows),
                !empty($_POST['latitude']) ? $_POST['latitude'] : null,
                !empty($_POST['longitude']) ? $_POST['longitude'] : null,
                $_POST['what3words'] ?? '',
                $_POST['google_maps_place_id'] ?? '',
                $_POST['google_maps_url'] ?? '',
                $_POST['google_street_view_url'] ?? '',
                json_encode($custom_fields)
            ]);
            
            $success = 'Location created successfully!';
            $location_id = $pdo->lastInsertId();
            
            // Redirect to view the new location
            header('Location: view-location.php?id=' . $location_id);
            exit;
        }
        
    } catch (Exception $e) {
        $error = 'Error saving location: ' . $e->getMessage();
        error_log($error);
    }
}

$page_title = $editing ? 'Edit Location' : 'Add New Location';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="https://maps.googleapis.com/maps/api/js?key=YOUR_GOOGLE_MAPS_API_KEY&libraries=places" async defer></script>
    <style>
        #map {
            width: 100%;
            height: 400px;
            border-radius: 8px;
            border: 2px solid var(--border-color);
        }
        
        .hours-grid {
            display: grid;
            grid-template-columns: 120px 1fr 1fr;
            gap: 0.5rem;
            align-items: center;
            margin-bottom: 0.75rem;
        }
        
        .delivery-window-row {
            display: grid;
            grid-template-columns: 1fr 1fr 2fr auto;
            gap: 0.75rem;
            margin-bottom: 0.75rem;
            align-items: end;
        }
        
        .coordinates-input {
            font-family: var(--font-mono);
            background: var(--bg-secondary);
        }
        
        .what3words-input {
            font-family: var(--font-mono);
            font-weight: 600;
        }
        
        .map-info {
            background: rgba(217, 119, 6, 0.1);
            border-left: 4px solid var(--primary);
            padding: 1rem;
            border-radius: 6px;
            margin-bottom: 1rem;
            font-size: 0.9rem;
            color: var(--text-secondary);
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <h1 class="page-title"><?php echo $page_title; ?></h1>
            <p class="page-subtitle">Complete delivery point information</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <span>✅</span>
                <span><?php echo sanitizeOutput($success); ?></span>
            </div>
        <?php endif; ?>

        <form method="POST" id="location-form">
            <!-- Basic Information -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">📋 Basic Information</h2>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="location_name">Location Name</label>
                    <input type="text" id="location_name" name="location_name" class="form-control" 
                           value="<?php echo sanitizeOutput($location['location_name'] ?? ''); ?>" required>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label required" for="address_line1">Address Line 1</label>
                        <input type="text" id="address_line1" name="address_line1" class="form-control" 
                               value="<?php echo sanitizeOutput($location['address_line1'] ?? ''); ?>" required>
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="address_line2">Address Line 2</label>
                        <input type="text" id="address_line2" name="address_line2" class="form-control" 
                               value="<?php echo sanitizeOutput($location['address_line2'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label required" for="city">City</label>
                        <input type="text" id="city" name="city" class="form-control" 
                               value="<?php echo sanitizeOutput($location['city'] ?? ''); ?>" required>
                    </div>

                    <div class="form-group">
                        <label class="form-label required" for="postcode">Postcode</label>
                        <input type="text" id="postcode" name="postcode" class="form-control" 
                               value="<?php echo sanitizeOutput($location['postcode'] ?? ''); ?>" required
                               style="text-transform: uppercase;" maxlength="10">
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="region">Region</label>
                        <input type="text" id="region" name="region" class="form-control" 
                               value="<?php echo sanitizeOutput($location['region'] ?? ''); ?>">
                    </div>
                </div>
            </div>

            <!-- Contact Information -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">📞 Contact Information</h2>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label" for="contact_name">Contact Name</label>
                        <input type="text" id="contact_name" name="contact_name" class="form-control" 
                               value="<?php echo sanitizeOutput($location['contact_name'] ?? ''); ?>">
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="contact_phone">Contact Phone</label>
                        <input type="tel" id="contact_phone" name="contact_phone" class="form-control" 
                               value="<?php echo sanitizeOutput($location['contact_phone'] ?? ''); ?>">
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="contact_email">Contact Email</label>
                        <input type="email" id="contact_email" name="contact_email" class="form-control" 
                               value="<?php echo sanitizeOutput($location['contact_email'] ?? ''); ?>">
                    </div>
                </div>
            </div>

            <!-- Opening Hours -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">🕐 Opening Hours</h2>
                </div>

                <?php 
                $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
                foreach ($days as $day): 
                    $hours = $location['opening_hours'][$day] ?? ['open' => '', 'close' => ''];
                ?>
                    <div class="hours-grid">
                        <label style="font-weight: 600; color: var(--text-primary);">
                            <?php echo ucfirst($day); ?>:
                        </label>
                        <input type="time" name="hours_<?php echo $day; ?>_open" class="form-control" 
                               value="<?php echo $hours['open']; ?>" placeholder="Opening">
                        <input type="time" name="hours_<?php echo $day; ?>_close" class="form-control" 
                               value="<?php echo $hours['close']; ?>" placeholder="Closing">
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Delivery Windows -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">🚚 Delivery Windows</h2>
                    <button type="button" class="btn btn-secondary" onclick="addDeliveryWindow()">
                        ➕ Add Window
                    </button>
                </div>

                <div id="delivery-windows-container">
                    <?php 
                    $windows = $location['delivery_windows'] ?? [];
                    if (empty($windows)) {
                        $windows = [['start' => '', 'end' => '', 'days' => '']];
                    }
                    foreach ($windows as $idx => $window): 
                    ?>
                        <div class="delivery-window-row">
                            <div class="form-group" style="margin-bottom: 0;">
                                <label class="form-label">Start Time</label>
                                <input type="time" name="delivery_window_start[]" class="form-control" 
                                       value="<?php echo $window['start']; ?>">
                            </div>
                            <div class="form-group" style="margin-bottom: 0;">
                                <label class="form-label">End Time</label>
                                <input type="time" name="delivery_window_end[]" class="form-control" 
                                       value="<?php echo $window['end']; ?>">
                            </div>
                            <div class="form-group" style="margin-bottom: 0;">
                                <label class="form-label">Days (e.g., Mon-Fri)</label>
                                <input type="text" name="delivery_window_days[]" class="form-control" 
                                       value="<?php echo sanitizeOutput($window['days'] ?? ''); ?>" 
                                       placeholder="All days">
                            </div>
                            <button type="button" class="btn btn-danger" onclick="this.parentElement.remove()" 
                                    style="padding: 0.75rem 1rem;">❌</button>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Location & Mapping -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">🗺️ Location & Mapping</h2>
                </div>

                <div class="map-info">
                    <strong>💡 Tip:</strong> Enter the address above and click "Find on Map" to automatically populate GPS coordinates and What3Words.
                    You can also click directly on the map to set the delivery point location.
                </div>

                <div class="form-row" style="margin-bottom: 1rem;">
                    <button type="button" class="btn btn-primary" onclick="geocodeAddress()">
                        🔍 Find on Map
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="getCurrentLocation()">
                        📍 Use Current Location
                    </button>
                </div>

                <div id="map" style="margin-bottom: 1.5rem;"></div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label" for="latitude">Latitude</label>
                        <input type="number" id="latitude" name="latitude" class="form-control coordinates-input" 
                               step="0.000001" min="-90" max="90"
                               value="<?php echo $location['latitude'] ?? ''; ?>" 
                               placeholder="51.5074">
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="longitude">Longitude</label>
                        <input type="number" id="longitude" name="longitude" class="form-control coordinates-input" 
                               step="0.000001" min="-180" max="180"
                               value="<?php echo $location['longitude'] ?? ''; ?>" 
                               placeholder="-0.1278">
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="what3words">
                            What3Words
                            <a href="https://what3words.com/" target="_blank" style="color: var(--primary); font-size: 0.85rem;">(lookup)</a>
                        </label>
                        <div style="position: relative;">
                            <span style="position: absolute; left: 1rem; top: 50%; transform: translateY(-50%); color: var(--danger); font-weight: 600;">//</span>
                            <input type="text" id="what3words" name="what3words" class="form-control what3words-input" 
                                   value="<?php echo sanitizeOutput($location['what3words'] ?? ''); ?>" 
                                   placeholder="filled.count.soap" style="padding-left: 2.5rem;">
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label" for="google_street_view_url">
                        Google Street View URL
                        <button type="button" class="btn btn-secondary" onclick="generateStreetViewUrl()" 
                                style="padding: 0.4rem 0.8rem; font-size: 0.85rem; margin-left: 0.5rem;">
                            🎥 Generate from Coordinates
                        </button>
                    </label>
                    <input type="url" id="google_street_view_url" name="google_street_view_url" class="form-control" 
                           value="<?php echo sanitizeOutput($location['google_street_view_url'] ?? ''); ?>" 
                           placeholder="https://www.google.com/maps/@?api=1&map_action=pano&viewpoint=...">
                    <small style="color: var(--text-muted); margin-top: 0.25rem; display: block;">
                        Direct link to Street View at this location. Auto-generated from GPS coordinates or enter custom URL.
                    </small>
                </div>

                <input type="hidden" id="google_maps_place_id" name="google_maps_place_id" 
                       value="<?php echo sanitizeOutput($location['google_maps_place_id'] ?? ''); ?>">
                <input type="hidden" id="google_maps_url" name="google_maps_url" 
                       value="<?php echo sanitizeOutput($location['google_maps_url'] ?? ''); ?>">
            </div>

            <!-- Custom Fields -->
            <?php if (!empty($custom_field_defs)): ?>
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">📝 Additional Information</h2>
                    <?php if ($_SESSION['role'] === 'admin'): ?>
                        <a href="custom-fields.php" class="btn btn-secondary" target="_blank">⚙️ Manage Fields</a>
                    <?php endif; ?>
                </div>

                <div class="form-row">
                    <?php foreach ($custom_field_defs as $field): 
                        $field_value = $location['custom_fields'][$field['field_name']] ?? '';
                    ?>
                        <div class="form-group">
                            <label class="form-label <?php echo $field['is_required'] ? 'required' : ''; ?>" 
                                   for="custom_<?php echo $field['field_name']; ?>">
                                <?php echo sanitizeOutput($field['field_label']); ?>
                            </label>
                            
                            <?php if ($field['field_type'] === 'textarea'): ?>
                                <textarea 
                                    id="custom_<?php echo $field['field_name']; ?>" 
                                    name="custom_field[<?php echo $field['field_name']; ?>]" 
                                    class="form-textarea" 
                                    rows="3"
                                    <?php echo $field['is_required'] ? 'required' : ''; ?>
                                    placeholder="<?php echo sanitizeOutput($field['help_text'] ?? ''); ?>"
                                ><?php echo sanitizeOutput($field_value); ?></textarea>
                            
                            <?php elseif ($field['field_type'] === 'select'): ?>
                                <select 
                                    id="custom_<?php echo $field['field_name']; ?>" 
                                    name="custom_field[<?php echo $field['field_name']; ?>]" 
                                    class="form-select"
                                    <?php echo $field['is_required'] ? 'required' : ''; ?>>
                                    <option value="">Select...</option>
                                    <?php 
                                    $options = json_decode($field['field_options'], true) ?: [];
                                    foreach ($options as $option): 
                                    ?>
                                        <option value="<?php echo sanitizeOutput($option); ?>" 
                                                <?php echo $field_value === $option ? 'selected' : ''; ?>>
                                            <?php echo sanitizeOutput($option); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            
                            <?php elseif ($field['field_type'] === 'checkbox'): ?>
                                <div style="display: flex; align-items: center; gap: 0.5rem; margin-top: 0.5rem;">
                                    <input 
                                        type="checkbox" 
                                        id="custom_<?php echo $field['field_name']; ?>" 
                                        name="custom_field[<?php echo $field['field_name']; ?>]" 
                                        value="1"
                                        <?php echo $field_value ? 'checked' : ''; ?>
                                        style="width: auto;">
                                    <label for="custom_<?php echo $field['field_name']; ?>" style="margin: 0;">
                                        <?php echo sanitizeOutput($field['help_text'] ?: 'Yes'); ?>
                                    </label>
                                </div>
                            
                            <?php else: ?>
                                <input 
                                    type="<?php echo $field['field_type']; ?>" 
                                    id="custom_<?php echo $field['field_name']; ?>" 
                                    name="custom_field[<?php echo $field['field_name']; ?>]" 
                                    class="form-control"
                                    value="<?php echo sanitizeOutput($field_value); ?>"
                                    <?php echo $field['is_required'] ? 'required' : ''; ?>
                                    placeholder="<?php echo sanitizeOutput($field['help_text'] ?? ''); ?>"
                                    <?php if ($field['field_type'] === 'number'): ?>
                                        step="0.01"
                                    <?php endif; ?>>
                            <?php endif; ?>
                            
                            <?php if ($field['help_text'] && $field['field_type'] !== 'checkbox'): ?>
                                <small style="color: var(--text-muted); margin-top: 0.25rem; display: block;">
                                    <?php echo sanitizeOutput($field['help_text']); ?>
                                </small>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <!-- Notes -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">📝 Additional Notes</h2>
                </div>

                <div class="form-group">
                    <label class="form-label" for="notes">Notes</label>
                    <textarea id="notes" name="notes" class="form-textarea" rows="4"
                              placeholder="Any additional information about this location..."><?php echo sanitizeOutput($location['notes'] ?? ''); ?></textarea>
                </div>
            </div>

            <div class="btn-group">
                <button type="submit" class="btn btn-primary">
                    💾 <?php echo $editing ? 'Update' : 'Create'; ?> Location
                </button>
                <a href="locations.php" class="btn btn-secondary">Cancel</a>
            </div>
        </form>
    </div>

    <script src="js/app.js"></script>
    <script>
        let map;
        let marker;
        let geocoder;
        
        // Initialize Google Map
        function initMap() {
            const defaultLat = parseFloat(document.getElementById('latitude').value) || 51.5074;
            const defaultLng = parseFloat(document.getElementById('longitude').value) || -0.1278;
            
            const center = { lat: defaultLat, lng: defaultLng };
            
            map = new google.maps.Map(document.getElementById('map'), {
                zoom: 15,
                center: center,
                mapTypeId: 'roadmap'
            });
            
            geocoder = new google.maps.Geocoder();
            
            // Add marker
            marker = new google.maps.Marker({
                position: center,
                map: map,
                draggable: true,
                title: 'Delivery Point'
            });
            
            // Update coordinates when marker is dragged
            marker.addListener('dragend', function() {
                const position = marker.getPosition();
                updateCoordinates(position.lat(), position.lng());
            });
            
            // Add click listener to map
            map.addListener('click', function(event) {
                marker.setPosition(event.latLng);
                updateCoordinates(event.latLng.lat(), event.latLng.lng());
            });
        }
        
        // Geocode address
        function geocodeAddress() {
            const address = [
                document.getElementById('address_line1').value,
                document.getElementById('address_line2').value,
                document.getElementById('city').value,
                document.getElementById('postcode').value
            ].filter(Boolean).join(', ');
            
            if (!address) {
                window.app.showAlert('Please enter an address first', 'warning');
                return;
            }
            
            geocoder.geocode({ address: address + ', UK' }, function(results, status) {
                if (status === 'OK') {
                    const location = results[0].geometry.location;
                    map.setCenter(location);
                    marker.setPosition(location);
                    updateCoordinates(location.lat(), location.lng());
                    
                    // Store place ID and URL
                    document.getElementById('google_maps_place_id').value = results[0].place_id;
                    document.getElementById('google_maps_url').value = 
                        `https://www.google.com/maps/place/?q=place_id:${results[0].place_id}`;
                    
                    window.app.showAlert('Location found on map!', 'success');
                } else {
                    window.app.showAlert('Could not find location. Please check the address.', 'error');
                }
            });
        }
        
        // Get current location
        function getCurrentLocation() {
            if (navigator.geolocation) {
                window.app.showAlert('Getting your location...', 'info');
                navigator.geolocation.getCurrentPosition(function(position) {
                    const lat = position.coords.latitude;
                    const lng = position.coords.longitude;
                    const location = new google.maps.LatLng(lat, lng);
                    
                    map.setCenter(location);
                    marker.setPosition(location);
                    updateCoordinates(lat, lng);
                    
                    window.app.showAlert('Location set to your current position', 'success');
                }, function(error) {
                    window.app.showAlert('Could not get your location: ' + error.message, 'error');
                });
            } else {
                window.app.showAlert('Geolocation is not supported by your browser', 'error');
            }
        }
        
        // Update coordinates and fetch What3Words
        function updateCoordinates(lat, lng) {
            document.getElementById('latitude').value = lat.toFixed(6);
            document.getElementById('longitude').value = lng.toFixed(6);
            
            // Fetch What3Words address
            fetchWhat3Words(lat, lng);
            
            // Generate Google Maps URL
            document.getElementById('google_maps_url').value = 
                `https://www.google.com/maps?q=${lat},${lng}`;
            
            // Auto-generate Street View URL
            generateStreetViewUrl();
        }
        
        // Generate Street View URL from coordinates
        function generateStreetViewUrl() {
            const lat = document.getElementById('latitude').value;
            const lng = document.getElementById('longitude').value;
            
            if (lat && lng) {
                const streetViewUrl = `https://www.google.com/maps/@?api=1&map_action=pano&viewpoint=${lat},${lng}`;
                document.getElementById('google_street_view_url').value = streetViewUrl;
                window.app.showAlert('Street View URL generated!', 'success');
            } else {
                window.app.showAlert('Please enter GPS coordinates first', 'warning');
            }
        }
        
        // Fetch What3Words address
        async function fetchWhat3Words(lat, lng) {
            // Note: This requires a What3Words API key
            // For now, we'll show a placeholder
            try {
                // In production, you would call:
                // const response = await fetch(`https://api.what3words.com/v3/convert-to-3wa?coordinates=${lat},${lng}&key=YOUR_API_KEY`);
                // const data = await response.json();
                // document.getElementById('what3words').value = data.words;
                
                // Placeholder - you'll need to add your own What3Words API integration
                console.log('What3Words lookup for:', lat, lng);
                window.app.showAlert('Manual What3Words entry required (API key not configured)', 'info');
            } catch (error) {
                console.error('What3Words error:', error);
            }
        }
        
        // Add delivery window
        function addDeliveryWindow() {
            const container = document.getElementById('delivery-windows-container');
            const row = document.createElement('div');
            row.className = 'delivery-window-row';
            row.innerHTML = `
                <div class="form-group" style="margin-bottom: 0;">
                    <label class="form-label">Start Time</label>
                    <input type="time" name="delivery_window_start[]" class="form-control">
                </div>
                <div class="form-group" style="margin-bottom: 0;">
                    <label class="form-label">End Time</label>
                    <input type="time" name="delivery_window_end[]" class="form-control">
                </div>
                <div class="form-group" style="margin-bottom: 0;">
                    <label class="form-label">Days (e.g., Mon-Fri)</label>
                    <input type="text" name="delivery_window_days[]" class="form-control" placeholder="All days">
                </div>
                <button type="button" class="btn btn-danger" onclick="this.parentElement.remove()" 
                        style="padding: 0.75rem 1rem;">❌</button>
            `;
            container.appendChild(row);
        }
        
        // Initialize map when page loads
        window.addEventListener('load', function() {
            if (typeof google !== 'undefined') {
                initMap();
            } else {
                window.app.showAlert('Google Maps failed to load. Please check your API key.', 'warning');
            }
        });
        
        // Auto-uppercase postcode
        document.getElementById('postcode').addEventListener('input', function() {
            this.value = this.value.toUpperCase();
        });
    </script>
</body>
</html>
