<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can manage custom fields
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'add':
                    $stmt = $pdo->prepare("
                        INSERT INTO custom_field_definitions 
                        (field_name, field_label, field_type, field_options, is_required, display_order, help_text)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $_POST['field_name'],
                        $_POST['field_label'],
                        $_POST['field_type'],
                        !empty($_POST['field_options']) ? $_POST['field_options'] : null,
                        isset($_POST['is_required']) ? 1 : 0,
                        $_POST['display_order'] ?? 999,
                        $_POST['help_text'] ?? ''
                    ]);
                    $success = 'Custom field added successfully!';
                    break;
                
                case 'update':
                    $stmt = $pdo->prepare("
                        UPDATE custom_field_definitions 
                        SET field_label = ?, field_type = ?, field_options = ?, 
                            is_required = ?, display_order = ?, help_text = ?, is_active = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([
                        $_POST['field_label'],
                        $_POST['field_type'],
                        !empty($_POST['field_options']) ? $_POST['field_options'] : null,
                        isset($_POST['is_required']) ? 1 : 0,
                        $_POST['display_order'],
                        $_POST['help_text'] ?? '',
                        isset($_POST['is_active']) ? 1 : 0,
                        $_POST['field_id']
                    ]);
                    $success = 'Custom field updated successfully!';
                    break;
                
                case 'delete':
                    $stmt = $pdo->prepare("DELETE FROM custom_field_definitions WHERE id = ?");
                    $stmt->execute([$_POST['field_id']]);
                    $success = 'Custom field deleted successfully!';
                    break;
            }
        } catch (PDOException $e) {
            $error = 'Database error: ' . $e->getMessage();
        }
    }
}

// Fetch all custom fields
$stmt = $pdo->query("SELECT * FROM custom_field_definitions ORDER BY display_order, field_label");
$custom_fields = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Custom Fields - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .field-preview {
            background: var(--bg-secondary);
            padding: 1rem;
            border-radius: 6px;
            margin-top: 0.5rem;
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">⚙️ Custom Field Manager</h1>
                <p class="page-subtitle">Configure additional fields for location details</p>
            </div>
            <button onclick="window.app.openModal('add-field-modal')" class="btn btn-primary">
                ➕ Add Custom Field
            </button>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <span>✅</span>
                <span><?php echo sanitizeOutput($success); ?></span>
            </div>
        <?php endif; ?>

        <!-- Info Box -->
        <div class="card mb-3" style="background: rgba(217, 119, 6, 0.1); border-left: 4px solid var(--primary);">
            <h3 style="color: var(--primary); margin-bottom: 1rem;">💡 About Custom Fields</h3>
            <p style="color: var(--text-secondary); line-height: 1.8; margin-bottom: 0;">
                Custom fields allow you to capture additional information specific to your delivery operations.
                Fields you create here will appear on the location add/edit forms. Examples: height restrictions,
                access codes, loading bay numbers, equipment availability, etc.
            </p>
        </div>

        <!-- Custom Fields List -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Configured Custom Fields (<?php echo count($custom_fields); ?>)</h2>
            </div>

            <?php if (empty($custom_fields)): ?>
                <div style="text-align: center; padding: 3rem; color: var(--text-muted);">
                    <div style="font-size: 3rem; margin-bottom: 1rem;">📝</div>
                    <p>No custom fields configured yet. Click "Add Custom Field" to get started.</p>
                </div>
            <?php else: ?>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Order</th>
                                <th>Field Name</th>
                                <th>Label</th>
                                <th>Type</th>
                                <th>Required</th>
                                <th>Active</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($custom_fields as $field): ?>
                                <tr>
                                    <td><?php echo $field['display_order']; ?></td>
                                    <td style="font-family: var(--font-mono); color: var(--primary);">
                                        <?php echo sanitizeOutput($field['field_name']); ?>
                                    </td>
                                    <td><strong><?php echo sanitizeOutput($field['field_label']); ?></strong></td>
                                    <td>
                                        <span class="badge" style="background: rgba(8, 145, 178, 0.2); color: #22D3EE;">
                                            <?php echo strtoupper($field['field_type']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($field['is_required']): ?>
                                            <span style="color: var(--danger);">✓ Yes</span>
                                        <?php else: ?>
                                            <span style="color: var(--text-muted);">No</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($field['is_active']): ?>
                                            <span class="badge badge-success">Active</span>
                                        <?php else: ?>
                                            <span class="badge" style="background: rgba(148, 163, 184, 0.2); color: var(--text-muted);">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="btn-group" style="gap: 0.5rem;">
                                            <button onclick="editField(<?php echo htmlspecialchars(json_encode($field)); ?>)" 
                                                    class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                                ✏️ Edit
                                            </button>
                                            <form method="POST" style="display: inline;" 
                                                  onsubmit="return confirm('Are you sure you want to delete this field? This cannot be undone.');">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="field_id" value="<?php echo $field['id']; ?>">
                                                <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem;">
                                                    🗑️ Delete
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Add/Edit Field Modal -->
    <div id="add-field-modal" class="modal">
        <div class="modal-content" style="max-width: 700px;">
            <div class="modal-header">
                <h2 class="modal-title" id="modal-title">Add Custom Field</h2>
                <button type="button" class="modal-close" onclick="window.app.closeModal('add-field-modal')">&times;</button>
            </div>

            <form method="POST" id="field-form">
                <input type="hidden" name="action" id="form-action" value="add">
                <input type="hidden" name="field_id" id="field-id">

                <div class="form-group">
                    <label class="form-label required" for="field_name">Field Name (Database Key)</label>
                    <input type="text" id="field_name" name="field_name" class="form-control" 
                           pattern="[a-z_]+" 
                           title="Lowercase letters and underscores only"
                           placeholder="e.g., loading_bay_number"
                           required>
                    <small style="color: var(--text-muted); margin-top: 0.25rem; display: block;">
                        Lowercase letters and underscores only. Cannot be changed after creation.
                    </small>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="field_label">Field Label (Display Name)</label>
                    <input type="text" id="field_label" name="field_label" class="form-control" 
                           placeholder="e.g., Loading Bay Number"
                           required>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label required" for="field_type">Field Type</label>
                        <select id="field_type" name="field_type" class="form-select" required onchange="updateFieldOptions()">
                            <option value="text">Text (Short)</option>
                            <option value="textarea">Text Area (Long)</option>
                            <option value="number">Number</option>
                            <option value="date">Date</option>
                            <option value="time">Time</option>
                            <option value="select">Select Dropdown</option>
                            <option value="checkbox">Checkbox (Yes/No)</option>
                            <option value="url">URL</option>
                            <option value="email">Email</option>
                            <option value="tel">Phone Number</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="display_order">Display Order</label>
                        <input type="number" id="display_order" name="display_order" class="form-control" 
                               value="999" min="0">
                    </div>
                </div>

                <div class="form-group" id="field-options-group" style="display: none;">
                    <label class="form-label" for="field_options">Select Options (JSON)</label>
                    <textarea id="field_options" name="field_options" class="form-textarea" rows="3"
                              placeholder='["Option 1", "Option 2", "Option 3"]'></textarea>
                    <small style="color: var(--text-muted); margin-top: 0.25rem; display: block;">
                        JSON array format for select dropdown options
                    </small>
                </div>

                <div class="form-group">
                    <label class="form-label" for="help_text">Help Text</label>
                    <input type="text" id="help_text" name="help_text" class="form-control" 
                           placeholder="Optional guidance for users">
                </div>

                <div class="form-row" style="grid-template-columns: 1fr 1fr;">
                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" id="is_required" name="is_required" style="width: auto;">
                        <label for="is_required" style="margin: 0; font-weight: 600;">Required Field</label>
                    </div>

                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" id="is_active" name="is_active" checked style="width: auto;">
                        <label for="is_active" style="margin: 0; font-weight: 600;">Active</label>
                    </div>
                </div>

                <div class="field-preview" id="field-preview" style="display: none;">
                    <strong style="color: var(--text-primary);">Preview:</strong>
                    <div id="preview-content" style="margin-top: 0.5rem;"></div>
                </div>

                <div class="btn-group" style="margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Save Field</button>
                    <button type="button" class="btn btn-secondary" onclick="window.app.closeModal('add-field-modal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script src="js/app.js"></script>
    <script>
        function updateFieldOptions() {
            const fieldType = document.getElementById('field_type').value;
            const optionsGroup = document.getElementById('field-options-group');
            
            if (fieldType === 'select') {
                optionsGroup.style.display = 'block';
            } else {
                optionsGroup.style.display = 'none';
            }
            
            updatePreview();
        }

        function updatePreview() {
            const label = document.getElementById('field_label').value;
            const type = document.getElementById('field_type').value;
            const helpText = document.getElementById('help_text').value;
            const required = document.getElementById('is_required').checked;
            
            if (!label) {
                document.getElementById('field-preview').style.display = 'none';
                return;
            }
            
            document.getElementById('field-preview').style.display = 'block';
            
            let previewHTML = `
                <div class="form-group">
                    <label class="form-label ${required ? 'required' : ''}">${label}</label>
            `;
            
            switch (type) {
                case 'textarea':
                    previewHTML += '<textarea class="form-control" rows="3" placeholder="Enter text..."></textarea>';
                    break;
                case 'select':
                    previewHTML += '<select class="form-select"><option>Select option...</option></select>';
                    break;
                case 'checkbox':
                    previewHTML += '<input type="checkbox" style="width: auto;">';
                    break;
                default:
                    previewHTML += `<input type="${type}" class="form-control" placeholder="Enter ${label.toLowerCase()}...">`;
            }
            
            if (helpText) {
                previewHTML += `<small style="color: var(--text-muted); margin-top: 0.25rem; display: block;">${helpText}</small>`;
            }
            
            previewHTML += '</div>';
            
            document.getElementById('preview-content').innerHTML = previewHTML;
        }

        function editField(field) {
            document.getElementById('modal-title').textContent = 'Edit Custom Field';
            document.getElementById('form-action').value = 'update';
            document.getElementById('field-id').value = field.id;
            
            document.getElementById('field_name').value = field.field_name;
            document.getElementById('field_name').readOnly = true; // Can't change field name
            document.getElementById('field_label').value = field.field_label;
            document.getElementById('field_type').value = field.field_type;
            document.getElementById('field_options').value = field.field_options || '';
            document.getElementById('display_order').value = field.display_order;
            document.getElementById('help_text').value = field.help_text || '';
            document.getElementById('is_required').checked = field.is_required == 1;
            document.getElementById('is_active').checked = field.is_active == 1;
            
            updateFieldOptions();
            window.app.openModal('add-field-modal');
        }

        // Reset form when opening for add
        document.querySelector('[onclick*="add-field-modal"]').addEventListener('click', function() {
            document.getElementById('modal-title').textContent = 'Add Custom Field';
            document.getElementById('field-form').reset();
            document.getElementById('form-action').value = 'add';
            document.getElementById('field-id').value = '';
            document.getElementById('field_name').readOnly = false;
            document.getElementById('is_active').checked = true;
            updateFieldOptions();
        });

        // Update preview on input
        document.getElementById('field_label').addEventListener('input', updatePreview);
        document.getElementById('field_type').addEventListener('change', updatePreview);
        document.getElementById('help_text').addEventListener('input', updatePreview);
        document.getElementById('is_required').addEventListener('change', updatePreview);
    </script>
</body>
</html>
