<?php
require_once '../includes/config.php';
requireLogin();

$assessment_id = $_GET['assessment_id'] ?? null;
$error = '';
$success = '';

if (!$assessment_id) {
    header('Location: assessments.php');
    exit;
}

// Get current assessment
$stmt = $pdo->prepare("
    SELECT a.*, l.location_name, av.version_number as current_version
    FROM assessments a
    JOIN locations l ON a.location_id = l.id
    LEFT JOIN assessment_versions av ON a.current_version_id = av.id
    WHERE a.id = ?
");
$stmt->execute([$assessment_id]);
$assessment = $stmt->fetch();

if (!$assessment) {
    header('Location: assessments.php');
    exit;
}

// Handle revision creation
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        $new_version = ($assessment['version_count'] ?? 1) + 1;
        $revision_type = $_POST['revision_type'] ?? 'revision';
        $revision_reason = $_POST['revision_reason'] ?? '';
        $changes_summary = $_POST['changes_summary'] ?? '';
        
        // Create new version
        $stmt = $pdo->prepare("
            INSERT INTO assessment_versions (
                assessment_id, version_number, assessment_number, location_id, assessor_id,
                assessment_date, assessment_time, weather_conditions, vehicle_type, delivery_volume,
                overall_risk_level, status, notes, recommendations,
                version_type, revision_reason, changes_summary, is_current, created_by
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, ?)
        ");
        
        $stmt->execute([
            $assessment_id,
            $new_version,
            $assessment['assessment_number'],
            $assessment['location_id'],
            $_SESSION['user_id'], // New assessor is current user
            $_POST['assessment_date'] ?? $assessment['assessment_date'],
            $_POST['assessment_time'] ?? $assessment['assessment_time'],
            $_POST['weather_conditions'] ?? $assessment['weather_conditions'],
            $_POST['vehicle_type'] ?? $assessment['vehicle_type'],
            $_POST['delivery_volume'] ?? $assessment['delivery_volume'],
            $_POST['overall_risk_level'] ?? $assessment['overall_risk_level'],
            'completed',
            $_POST['notes'] ?? $assessment['notes'],
            $_POST['recommendations'] ?? $assessment['recommendations'],
            $revision_type,
            $revision_reason,
            $changes_summary,
            $_SESSION['user_id']
        ]);
        
        $new_version_id = $pdo->lastInsertId();
        
        // Mark old version as superseded
        if ($assessment['current_version_id']) {
            $stmt = $pdo->prepare("
                UPDATE assessment_versions 
                SET is_current = 0, status = 'superseded', supersedes_version = ?
                WHERE id = ?
            ");
            $stmt->execute([$new_version_id, $assessment['current_version_id']]);
        }
        
        // Update assessment
        $stmt = $pdo->prepare("
            UPDATE assessments 
            SET current_version_id = ?, 
                version_count = ?,
                last_revised_at = NOW(),
                last_revised_by = ?
            WHERE id = ?
        ");
        $stmt->execute([$new_version_id, $new_version, $_SESSION['user_id'], $assessment_id]);
        
        // Copy or update assessment details based on user selection
        if (isset($_POST['copy_details']) && $_POST['copy_details'] === 'yes') {
            // Copy existing details
            $stmt = $pdo->prepare("
                INSERT INTO assessment_version_details (
                    version_id, risk_item_id, likelihood, severity, risk_score,
                    current_controls, additional_actions, action_owner, target_date
                )
                SELECT ?, risk_item_id, likelihood, severity, risk_score,
                       current_controls, additional_actions, action_owner, target_date
                FROM assessment_details
                WHERE assessment_id = ?
            ");
            $stmt->execute([$new_version_id, $assessment_id]);
        }
        
        // Log the change
        $stmt = $pdo->prepare("
            INSERT INTO assessment_change_log (
                assessment_id, from_version, to_version, action_type, change_description, changed_by
            ) VALUES (?, ?, ?, 'revised', ?, ?)
        ");
        $stmt->execute([
            $assessment_id,
            $assessment['current_version_id'],
            $new_version_id,
            $revision_reason,
            $_SESSION['user_id']
        ]);
        
        $pdo->commit();
        
        // Redirect to edit the new revision
        header('Location: edit-assessment.php?id=' . $assessment_id . '&version=' . $new_version_id . '&success=revision_created');
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = 'Error creating revision: ' . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Revision - <?php echo sanitizeOutput($assessment['assessment_number']); ?></title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">➕ Create New Revision</h1>
                <p class="page-subtitle">
                    <?php echo sanitizeOutput($assessment['assessment_number']); ?> - 
                    <?php echo sanitizeOutput($assessment['location_name']); ?>
                </p>
            </div>
            <a href="version-history.php?id=<?php echo $assessment_id; ?>" class="btn btn-secondary">
                ← Back to History
            </a>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Revision Details</h2>
            </div>

            <form method="POST" style="padding: 1.5rem;">
                <!-- Current Version Info -->
                <div style="background: var(--bg-tertiary); padding: 1.5rem; border-radius: 8px; margin-bottom: 2rem;">
                    <h3 style="color: var(--text-primary); margin-bottom: 1rem;">Current Version</h3>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
                        <div>
                            <strong style="color: var(--text-muted); display: block; font-size: 0.85rem;">Version Number</strong>
                            <span style="color: var(--text-primary);">v<?php echo $assessment['current_version'] ?? 1; ?></span>
                        </div>
                        <div>
                            <strong style="color: var(--text-muted); display: block; font-size: 0.85rem;">Current Risk Level</strong>
                            <span style="color: var(--text-primary);"><?php echo strtoupper($assessment['overall_risk_level']); ?></span>
                        </div>
                        <div>
                            <strong style="color: var(--text-muted); display: block; font-size: 0.85rem;">Status</strong>
                            <span style="color: var(--text-primary);"><?php echo ucfirst($assessment['status']); ?></span>
                        </div>
                    </div>
                </div>

                <!-- Revision Type -->
                <div class="form-group">
                    <label class="form-label required">Revision Type</label>
                    <select name="revision_type" class="form-select" required>
                        <option value="revision">Minor Revision - Small updates or corrections</option>
                        <option value="major_revision">Major Revision - Significant changes to assessment</option>
                    </select>
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        Choose the type of revision based on the extent of changes
                    </small>
                </div>

                <!-- Revision Reason -->
                <div class="form-group">
                    <label class="form-label required">Reason for Revision</label>
                    <textarea name="revision_reason" class="form-textarea" rows="3" required
                              placeholder="Explain why this revision is being created..."></textarea>
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        This will be visible in the version history
                    </small>
                </div>

                <!-- Changes Summary -->
                <div class="form-group">
                    <label class="form-label">Summary of Changes</label>
                    <textarea name="changes_summary" class="form-textarea" rows="4"
                              placeholder="Summarize the changes being made in this revision..."></textarea>
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        Optional: Provide a brief summary of what's being changed
                    </small>
                </div>

                <!-- Copy Options -->
                <div class="form-group">
                    <label class="form-label">Data Handling</label>
                    <div style="background: var(--bg-tertiary); padding: 1.5rem; border-radius: 8px;">
                        <div style="margin-bottom: 1rem;">
                            <label style="display: flex; align-items: start; gap: 0.75rem; cursor: pointer;">
                                <input type="radio" name="copy_details" value="yes" checked style="margin-top: 0.25rem;">
                                <div>
                                    <strong style="display: block; color: var(--text-primary); margin-bottom: 0.25rem;">
                                        Copy Existing Risk Items
                                    </strong>
                                    <span style="color: var(--text-muted); font-size: 0.85rem;">
                                        Start with current risk items and ratings, then modify as needed
                                    </span>
                                </div>
                            </label>
                        </div>
                        
                        <div>
                            <label style="display: flex; align-items: start; gap: 0.75rem; cursor: pointer;">
                                <input type="radio" name="copy_details" value="no" style="margin-top: 0.25rem;">
                                <div>
                                    <strong style="display: block; color: var(--text-primary); margin-bottom: 0.25rem;">
                                        Start Fresh
                                    </strong>
                                    <span style="color: var(--text-muted); font-size: 0.85rem;">
                                        Begin with empty risk items for complete re-assessment
                                    </span>
                                </div>
                            </label>
                        </div>
                    </div>
                </div>

                <!-- Update Basic Info (Optional) -->
                <div style="border-top: 1px solid var(--border-color); margin: 2rem 0; padding-top: 2rem;">
                    <h3 style="color: var(--text-primary); margin-bottom: 1rem;">
                        Update Assessment Information (Optional)
                    </h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Assessment Date</label>
                            <input type="date" name="assessment_date" class="form-control"
                                   value="<?php echo $assessment['assessment_date']; ?>">
                        </div>

                        <div class="form-group">
                            <label class="form-label">Time</label>
                            <input type="time" name="assessment_time" class="form-control"
                                   value="<?php echo $assessment['assessment_time']; ?>">
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">Weather Conditions</label>
                            <input type="text" name="weather_conditions" class="form-control"
                                   value="<?php echo sanitizeOutput($assessment['weather_conditions']); ?>">
                        </div>

                        <div class="form-group">
                            <label class="form-label">Vehicle Type</label>
                            <select name="vehicle_type" class="form-select">
                                <option value="">Select...</option>
                                <?php
                                $stmt = $pdo->query("SELECT type_name FROM vehicle_types WHERE is_active = 1 ORDER BY display_order");
                                while ($type = $stmt->fetch()):
                                ?>
                                    <option value="<?php echo $type['type_name']; ?>" 
                                            <?php echo $type['type_name'] === $assessment['vehicle_type'] ? 'selected' : ''; ?>>
                                        <?php echo sanitizeOutput($type['type_name']); ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="btn-group" style="margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">
                        ➕ Create Revision & Continue Editing
                    </button>
                    <a href="version-history.php?id=<?php echo $assessment_id; ?>" class="btn btn-secondary">
                        Cancel
                    </a>
                </div>

                <div style="margin-top: 1.5rem; padding: 1rem; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px;">
                    <strong style="color: #856404;">ℹ️ Note:</strong>
                    <p style="color: #856404; margin: 0.5rem 0 0 0;">
                        Creating a revision will generate a new version (v<?php echo ($assessment['version_count'] ?? 1) + 1; ?>) 
                        and mark the current version as superseded. The original data is preserved and can be viewed in the version history.
                    </p>
                </div>
            </form>
        </div>
    </div>

    <script src="js/app.js"></script>
</body>
</html>
