<?php
require_once '../includes/config.php';
requireLogin();

$v1_id = $_GET['v1'] ?? null;
$v2_id = $_GET['v2'] ?? null;

if (!$v1_id || !$v2_id) {
    header('Location: assessments.php');
    exit;
}

// Load both versions
$stmt = $pdo->prepare("
    SELECT av.*, l.location_name, u.full_name as assessor_name
    FROM assessment_versions av
    JOIN locations l ON av.location_id = l.id
    JOIN users u ON av.assessor_id = u.id
    WHERE av.id = ?
");

$stmt->execute([$v1_id]);
$v1 = $stmt->fetch();

$stmt->execute([$v2_id]);
$v2 = $stmt->fetch();

if (!$v1 || !$v2 || $v1['assessment_id'] !== $v2['assessment_id']) {
    header('Location: assessments.php');
    exit;
}

// Load details for both
$stmt = $pdo->prepare("
    SELECT avd.*, ri.item_name, rc.category_name
    FROM assessment_version_details avd
    JOIN risk_items ri ON avd.risk_item_id = ri.id
    JOIN risk_categories rc ON ri.category_id = rc.id
    WHERE avd.version_id = ?
    ORDER BY rc.display_order, ri.display_order
");

$stmt->execute([$v1_id]);
$v1_details = $stmt->fetchAll();

$stmt->execute([$v2_id]);
$v2_details = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0;">
    <title>Compare Versions</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .comparison-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
        }
        
        .version-column {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 8px;
            padding: 1.5rem;
        }
        
        .diff-highlight {
            background: #fef3c7;
            padding: 0.125rem 0.25rem;
            border-radius: 3px;
        }
        
        .field-comparison {
            margin-bottom: 1rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .field-comparison:last-child {
            border-bottom: none;
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">🔄 Compare Versions</h1>
                <p class="page-subtitle">
                    <?php echo sanitizeOutput($v1['assessment_number']); ?> - 
                    <?php echo sanitizeOutput($v1['location_name']); ?>
                </p>
            </div>
            <a href="version-history.php?id=<?php echo $v1['assessment_id']; ?>" class="btn btn-secondary">
                ← Back to History
            </a>
        </div>

        <!-- Version Headers -->
        <div class="comparison-grid" style="margin-bottom: 1.5rem;">
            <div class="version-column">
                <h2 style="color: var(--text-primary); margin-bottom: 0.5rem;">
                    Version <?php echo $v1['version_number']; ?>
                </h2>
                <div style="color: var(--text-muted); font-size: 0.9rem;">
                    <?php echo date('d/m/Y H:i', strtotime($v1['created_at'])); ?><br>
                    By <?php echo sanitizeOutput($v1['assessor_name']); ?>
                </div>
            </div>
            
            <div class="version-column">
                <h2 style="color: var(--text-primary); margin-bottom: 0.5rem;">
                    Version <?php echo $v2['version_number']; ?>
                </h2>
                <div style="color: var(--text-muted); font-size: 0.9rem;">
                    <?php echo date('d/m/Y H:i', strtotime($v2['created_at'])); ?><br>
                    By <?php echo sanitizeOutput($v2['assessor_name']); ?>
                </div>
            </div>
        </div>

        <!-- Basic Info Comparison -->
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">Assessment Information</h2>
            </div>

            <div style="padding: 1.5rem;">
                <?php
                $fields = [
                    ['label' => 'Date', 'key' => 'assessment_date'],
                    ['label' => 'Time', 'key' => 'assessment_time'],
                    ['label' => 'Weather', 'key' => 'weather_conditions'],
                    ['label' => 'Vehicle Type', 'key' => 'vehicle_type'],
                    ['label' => 'Delivery Volume', 'key' => 'delivery_volume'],
                    ['label' => 'Overall Risk Level', 'key' => 'overall_risk_level'],
                    ['label' => 'Status', 'key' => 'status']
                ];
                
                foreach ($fields as $field):
                    $changed = $v1[$field['key']] !== $v2[$field['key']];
                ?>
                    <div class="field-comparison">
                        <strong style="color: var(--text-muted); display: block; margin-bottom: 0.5rem;">
                            <?php echo $field['label']; ?>:
                        </strong>
                        <div class="comparison-grid">
                            <div class="<?php echo $changed ? 'diff-highlight' : ''; ?>">
                                <?php echo sanitizeOutput($v1[$field['key']] ?? 'N/A'); ?>
                            </div>
                            <div class="<?php echo $changed ? 'diff-highlight' : ''; ?>">
                                <?php echo sanitizeOutput($v2[$field['key']] ?? 'N/A'); ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- Risk Items Comparison -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Risk Items Comparison</h2>
            </div>

            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Risk Item</th>
                            <th colspan="2" style="text-align: center; background: var(--bg-tertiary);">
                                v<?php echo $v1['version_number']; ?> → v<?php echo $v2['version_number']; ?>
                            </th>
                        </tr>
                        <tr>
                            <th></th>
                            <th>L × S = Score</th>
                            <th>L × S = Score</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        // Combine risk items
                        $all_items = [];
                        foreach ($v1_details as $d) {
                            $all_items[$d['risk_item_id']] = ['v1' => $d, 'v2' => null, 'name' => $d['item_name']];
                        }
                        foreach ($v2_details as $d) {
                            if (isset($all_items[$d['risk_item_id']])) {
                                $all_items[$d['risk_item_id']]['v2'] = $d;
                            } else {
                                $all_items[$d['risk_item_id']] = ['v1' => null, 'v2' => $d, 'name' => $d['item_name']];
                            }
                        }
                        
                        foreach ($all_items as $item):
                            $v1_data = $item['v1'];
                            $v2_data = $item['v2'];
                            $changed = false;
                            
                            if ($v1_data && $v2_data) {
                                $changed = $v1_data['risk_score'] !== $v2_data['risk_score'];
                            }
                        ?>
                            <tr>
                                <td><strong><?php echo sanitizeOutput($item['name']); ?></strong></td>
                                <td class="<?php echo $changed ? 'diff-highlight' : ''; ?>">
                                    <?php if ($v1_data): ?>
                                        <?php echo $v1_data['likelihood']; ?> × <?php echo $v1_data['severity']; ?> = 
                                        <strong><?php echo $v1_data['risk_score']; ?></strong>
                                    <?php else: ?>
                                        <em style="color: var(--text-muted);">Not assessed</em>
                                    <?php endif; ?>
                                </td>
                                <td class="<?php echo $changed ? 'diff-highlight' : ''; ?>">
                                    <?php if ($v2_data): ?>
                                        <?php echo $v2_data['likelihood']; ?> × <?php echo $v2_data['severity']; ?> = 
                                        <strong><?php echo $v2_data['risk_score']; ?></strong>
                                    <?php else: ?>
                                        <em style="color: var(--text-muted);">Not assessed</em>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script src="js/app.js"></script>
</body>
</html>
