<?php
/**
 * Photo Upload API
 * Handles photo uploads for assessments with captions
 */

require_once '../../includes/config.php';
requireLogin();

header('Content-Type: application/json');

// Configuration
define('UPLOAD_DIR', __DIR__ . '/../../uploads/photos/');
define('MAX_FILE_SIZE', 10 * 1024 * 1024); // 10MB
define('ALLOWED_TYPES', ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp']);

// Ensure upload directory exists
if (!file_exists(UPLOAD_DIR)) {
    mkdir(UPLOAD_DIR, 0755, true);
}

// Handle different request types
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'POST':
            handleUpload();
            break;
        
        case 'DELETE':
            handleDelete();
            break;
        
        case 'PUT':
            handleUpdate();
            break;
        
        case 'GET':
            handleGet();
            break;
        
        default:
            throw new Exception('Method not allowed');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function handleUpload() {
    global $pdo;
    
    // Validate input
    if (!isset($_FILES['photo'])) {
        throw new Exception('No file uploaded');
    }
    
    if (!isset($_POST['assessment_id'])) {
        throw new Exception('Assessment ID required');
    }
    
    $file = $_FILES['photo'];
    $assessment_id = (int)$_POST['assessment_id'];
    $caption = $_POST['caption'] ?? '';
    $display_order = $_POST['display_order'] ?? 0;
    
    // Validate file
    if ($file['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('Upload error: ' . $file['error']);
    }
    
    if ($file['size'] > MAX_FILE_SIZE) {
        throw new Exception('File too large. Maximum size is 10MB');
    }
    
    if (!in_array($file['type'], ALLOWED_TYPES)) {
        throw new Exception('Invalid file type. Only JPG, PNG, GIF, and WebP images are allowed');
    }
    
    // Verify assessment exists and user has access
    $stmt = $pdo->prepare("SELECT id FROM assessments WHERE id = ?");
    $stmt->execute([$assessment_id]);
    if (!$stmt->fetch()) {
        throw new Exception('Assessment not found');
    }
    
    // Generate unique filename
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid('photo_' . $assessment_id . '_', true) . '.' . $extension;
    $filepath = UPLOAD_DIR . $filename;
    
    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $filepath)) {
        throw new Exception('Failed to save file');
    }
    
    // Optimize/resize image if needed
    optimizeImage($filepath, $file['type']);
    
    // Save to database
    $stmt = $pdo->prepare("
        INSERT INTO photos (assessment_id, filename, original_filename, file_path, 
                          file_size, mime_type, caption, display_order, uploaded_by)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $assessment_id,
        $filename,
        $file['name'],
        'uploads/photos/' . $filename,
        filesize($filepath),
        $file['type'],
        $caption,
        $display_order,
        $_SESSION['user_id']
    ]);
    
    $photo_id = $pdo->lastInsertId();
    
    // Return success with photo data
    echo json_encode([
        'success' => true,
        'photo' => [
            'id' => $photo_id,
            'filename' => $filename,
            'original_filename' => $file['name'],
            'file_path' => 'uploads/photos/' . $filename,
            'caption' => $caption,
            'file_size' => filesize($filepath),
            'display_order' => $display_order
        ]
    ]);
}

function handleDelete() {
    global $pdo;
    
    parse_str(file_get_contents("php://input"), $_DELETE);
    
    if (!isset($_DELETE['photo_id'])) {
        throw new Exception('Photo ID required');
    }
    
    $photo_id = (int)$_DELETE['photo_id'];
    
    // Get photo details
    $stmt = $pdo->prepare("SELECT filename, assessment_id FROM photos WHERE id = ?");
    $stmt->execute([$photo_id]);
    $photo = $stmt->fetch();
    
    if (!$photo) {
        throw new Exception('Photo not found');
    }
    
    // Verify user has access to this assessment
    $stmt = $pdo->prepare("SELECT assessor_id FROM assessments WHERE id = ?");
    $stmt->execute([$photo['assessment_id']]);
    $assessment = $stmt->fetch();
    
    if ($_SESSION['role'] !== 'admin' && $_SESSION['user_id'] != $assessment['assessor_id']) {
        throw new Exception('Access denied');
    }
    
    // Delete file
    $filepath = UPLOAD_DIR . $photo['filename'];
    if (file_exists($filepath)) {
        unlink($filepath);
    }
    
    // Delete from database
    $stmt = $pdo->prepare("DELETE FROM photos WHERE id = ?");
    $stmt->execute([$photo_id]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Photo deleted successfully'
    ]);
}

function handleUpdate() {
    global $pdo;
    
    parse_str(file_get_contents("php://input"), $_PUT);
    
    if (!isset($_PUT['photo_id'])) {
        throw new Exception('Photo ID required');
    }
    
    $photo_id = (int)$_PUT['photo_id'];
    $caption = $_PUT['caption'] ?? '';
    $display_order = $_PUT['display_order'] ?? 0;
    
    // Update database
    $stmt = $pdo->prepare("
        UPDATE photos 
        SET caption = ?, display_order = ?
        WHERE id = ?
    ");
    
    $stmt->execute([$caption, $display_order, $photo_id]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Photo updated successfully'
    ]);
}

function handleGet() {
    global $pdo;
    
    if (!isset($_GET['assessment_id'])) {
        throw new Exception('Assessment ID required');
    }
    
    $assessment_id = (int)$_GET['assessment_id'];
    
    // Fetch all photos for this assessment
    $stmt = $pdo->prepare("
        SELECT p.*, u.full_name as uploaded_by_name
        FROM photos p
        LEFT JOIN users u ON p.uploaded_by = u.id
        WHERE p.assessment_id = ?
        ORDER BY p.display_order, p.uploaded_at
    ");
    
    $stmt->execute([$assessment_id]);
    $photos = $stmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'photos' => $photos
    ]);
}

function optimizeImage($filepath, $mime_type) {
    // Simple image optimization
    // You can extend this with more sophisticated optimization
    
    $max_width = 1920;
    $max_height = 1920;
    $quality = 85;
    
    list($width, $height) = getimagesize($filepath);
    
    // Only optimize if image is larger than max dimensions
    if ($width <= $max_width && $height <= $max_height) {
        return;
    }
    
    // Calculate new dimensions
    $ratio = min($max_width / $width, $max_height / $height);
    $new_width = round($width * $ratio);
    $new_height = round($height * $ratio);
    
    // Create new image
    $source = null;
    switch ($mime_type) {
        case 'image/jpeg':
        case 'image/jpg':
            $source = imagecreatefromjpeg($filepath);
            break;
        case 'image/png':
            $source = imagecreatefrompng($filepath);
            break;
        case 'image/gif':
            $source = imagecreatefromgif($filepath);
            break;
        case 'image/webp':
            $source = imagecreatefromwebp($filepath);
            break;
    }
    
    if (!$source) {
        return;
    }
    
    $destination = imagecreatetruecolor($new_width, $new_height);
    
    // Preserve transparency for PNG
    if ($mime_type === 'image/png') {
        imagealphablending($destination, false);
        imagesavealpha($destination, true);
    }
    
    imagecopyresampled($destination, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
    
    // Save optimized image
    switch ($mime_type) {
        case 'image/jpeg':
        case 'image/jpg':
            imagejpeg($destination, $filepath, $quality);
            break;
        case 'image/png':
            imagepng($destination, $filepath, 9);
            break;
        case 'image/gif':
            imagegif($destination, $filepath);
            break;
        case 'image/webp':
            imagewebp($destination, $filepath, $quality);
            break;
    }
    
    imagedestroy($source);
    imagedestroy($destination);
}
