<?php
/**
 * Simple PDF Export API (PHP-based, no Python required)
 * Generates basic PDF reports using TCPDF or fallback to HTML
 */

require_once '../../includes/config.php';
requireLogin();

header('Content-Type: application/json');

if (!isset($_GET['assessment_id'])) {
    echo json_encode(['success' => false, 'error' => 'Assessment ID required']);
    exit;
}

$assessment_id = (int)$_GET['assessment_id'];

try {
    // Fetch complete assessment data
    $stmt = $pdo->prepare("
        SELECT a.*, l.location_name, l.address_line1, l.address_line2, l.city, l.postcode, 
               l.contact_name, l.contact_phone,
               u.full_name as assessor_name, u.email as assessor_email
        FROM assessments a
        JOIN locations l ON a.location_id = l.id
        JOIN users u ON a.assessor_id = u.id
        WHERE a.id = ?
    ");
    $stmt->execute([$assessment_id]);
    $assessment = $stmt->fetch();
    
    if (!$assessment) {
        throw new Exception('Assessment not found');
    }
    
    // Fetch assessment details
    $stmt = $pdo->prepare("
        SELECT ad.*, ri.item_name, rc.category_name
        FROM assessment_details ad
        JOIN risk_items ri ON ad.risk_item_id = ri.id
        JOIN risk_categories rc ON ri.category_id = rc.id
        WHERE ad.assessment_id = ?
        ORDER BY rc.display_order, ri.display_order
    ");
    $stmt->execute([$assessment_id]);
    $details = $stmt->fetchAll();
    
    // Group by category
    $grouped_details = [];
    foreach ($details as $detail) {
        $grouped_details[$detail['category_name']][] = $detail;
    }
    
    // Generate HTML content
    $html = generatePDFHTML($assessment, $grouped_details);
    
    // Create reports directory if needed
    $reports_dir = __DIR__ . '/../../reports';
    if (!file_exists($reports_dir)) {
        mkdir($reports_dir, 0755, true);
    }
    
    // Generate filename
    $filename = 'assessment_' . $assessment['assessment_number'] . '_' . date('Ymd_His') . '.html';
    $output_path = $reports_dir . '/' . $filename;
    
    // Save HTML file
    file_put_contents($output_path, $html);
    
    // Return success
    echo json_encode([
        'success' => true,
        'filename' => $filename,
        'download_url' => 'api/download-report.php?file=' . urlencode($filename),
        'print_url' => 'api/print-report.php?file=' . urlencode($filename),
        'message' => 'Report generated successfully'
    ]);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function generatePDFHTML($assessment, $grouped_details) {
    ob_start();
    ?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Risk Assessment Report - <?php echo htmlspecialchars($assessment['assessment_number']); ?></title>
    <style>
        @media print {
            @page { margin: 2cm; }
            body { margin: 0; }
        }
        
        * { box-sizing: border-box; }
        
        body {
            font-family: 'Helvetica', 'Arial', sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 210mm;
            margin: 0 auto;
            padding: 20px;
            background: white;
        }
        
        .header {
            text-align: center;
            border-bottom: 4px solid #D97706;
            padding-bottom: 20px;
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: #D97706;
            font-size: 28px;
            margin: 0 0 10px 0;
        }
        
        .header .subtitle {
            color: #666;
            font-size: 16px;
        }
        
        .section {
            margin-bottom: 30px;
            page-break-inside: avoid;
        }
        
        .section-title {
            background: #D97706;
            color: white;
            padding: 10px 15px;
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 15px;
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: 150px 1fr;
            gap: 10px;
            margin-bottom: 10px;
        }
        
        .info-label {
            font-weight: bold;
            color: #555;
        }
        
        .info-value {
            color: #333;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        
        table th {
            background: #334155;
            color: white;
            padding: 12px 10px;
            text-align: left;
            font-weight: bold;
            font-size: 14px;
        }
        
        table td {
            padding: 10px;
            border-bottom: 1px solid #ddd;
            font-size: 13px;
        }
        
        table tr:nth-child(even) {
            background: #f8f9fa;
        }
        
        .risk-low { color: #059669; font-weight: bold; }
        .risk-medium { color: #F59E0B; font-weight: bold; }
        .risk-high { color: #DC2626; font-weight: bold; }
        .risk-critical { color: #991B1B; font-weight: bold; }
        
        .footer {
            margin-top: 40px;
            padding-top: 20px;
            border-top: 2px solid #ddd;
            text-align: center;
            color: #666;
            font-size: 12px;
        }
        
        .category-header {
            background: #F3F4F6;
            padding: 8px 15px;
            font-weight: bold;
            font-size: 16px;
            color: #374151;
            margin-top: 20px;
            margin-bottom: 10px;
        }
        
        @media print {
            .no-print { display: none; }
            body { padding: 0; }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>⚠️ Risk Assessment Report</h1>
        <div class="subtitle">Supermarket Delivery Risk Assessment</div>
    </div>
    
    <!-- Assessment Information -->
    <div class="section">
        <div class="section-title">Assessment Information</div>
        <div class="info-grid">
            <div class="info-label">Assessment Number:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['assessment_number']); ?></div>
            
            <div class="info-label">Date:</div>
            <div class="info-value"><?php echo date('d/m/Y', strtotime($assessment['assessment_date'])); ?></div>
            
            <div class="info-label">Time:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['assessment_time']); ?></div>
            
            <div class="info-label">Assessor:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['assessor_name']); ?></div>
            
            <div class="info-label">Overall Risk Level:</div>
            <div class="info-value risk-<?php echo $assessment['overall_risk_level']; ?>">
                <?php echo strtoupper($assessment['overall_risk_level']); ?>
            </div>
            
            <div class="info-label">Status:</div>
            <div class="info-value"><?php echo ucfirst($assessment['status']); ?></div>
        </div>
    </div>
    
    <!-- Location Information -->
    <div class="section">
        <div class="section-title">Location Details</div>
        <div class="info-grid">
            <div class="info-label">Location Name:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['location_name']); ?></div>
            
            <div class="info-label">Address:</div>
            <div class="info-value">
                <?php 
                echo htmlspecialchars($assessment['address_line1']); 
                if (!empty($assessment['address_line2'])) {
                    echo '<br>' . htmlspecialchars($assessment['address_line2']);
                }
                ?>
            </div>
            
            <div class="info-label">City:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['city']); ?></div>
            
            <div class="info-label">Postcode:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['postcode']); ?></div>
            
            <?php if ($assessment['contact_name']): ?>
            <div class="info-label">Contact:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['contact_name']); ?></div>
            <?php endif; ?>
            
            <?php if ($assessment['contact_phone']): ?>
            <div class="info-label">Phone:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['contact_phone']); ?></div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Assessment Details -->
    <div class="section">
        <div class="section-title">Risk Assessment Details</div>
        <div class="info-grid">
            <div class="info-label">Weather:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['weather_conditions'] ?? 'N/A'); ?></div>
            
            <div class="info-label">Vehicle Type:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['vehicle_type'] ?? 'N/A'); ?></div>
            
            <div class="info-label">Delivery Volume:</div>
            <div class="info-value"><?php echo htmlspecialchars($assessment['delivery_volume'] ?? 'N/A'); ?></div>
        </div>
    </div>
    
    <!-- Risk Items by Category -->
    <?php if (!empty($grouped_details)): ?>
    <div class="section">
        <div class="section-title">Risk Evaluation</div>
        
        <?php foreach ($grouped_details as $category => $items): ?>
            <div class="category-header"><?php echo htmlspecialchars($category); ?></div>
            
            <table>
                <thead>
                    <tr>
                        <th style="width: 40%;">Risk Item</th>
                        <th style="width: 10%; text-align: center;">L</th>
                        <th style="width: 10%; text-align: center;">S</th>
                        <th style="width: 10%; text-align: center;">Score</th>
                        <th style="width: 30%;">Current Controls</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $item): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($item['item_name']); ?></td>
                        <td style="text-align: center;"><?php echo $item['likelihood']; ?></td>
                        <td style="text-align: center;"><?php echo $item['severity']; ?></td>
                        <td style="text-align: center;"><strong><?php echo $item['risk_score']; ?></strong></td>
                        <td><?php echo htmlspecialchars($item['current_controls'] ?? '-'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endforeach; ?>
    </div>
    <?php endif; ?>
    
    <!-- Notes and Recommendations -->
    <?php if ($assessment['notes'] || $assessment['recommendations']): ?>
    <div class="section">
        <div class="section-title">Notes & Recommendations</div>
        
        <?php if ($assessment['notes']): ?>
        <div style="margin-bottom: 15px;">
            <strong>Notes:</strong><br>
            <?php echo nl2br(htmlspecialchars($assessment['notes'])); ?>
        </div>
        <?php endif; ?>
        
        <?php if ($assessment['recommendations']): ?>
        <div>
            <strong>Recommendations:</strong><br>
            <?php echo nl2br(htmlspecialchars($assessment['recommendations'])); ?>
        </div>
        <?php endif; ?>
    </div>
    <?php endif; ?>
    
    <div class="footer">
        <p>Generated on <?php echo date('d/m/Y H:i'); ?></p>
        <p>Risk Assessment System - Supermarket Delivery Safety</p>
    </div>
    
    <div class="no-print" style="margin-top: 30px; text-align: center;">
        <button onclick="window.print()" style="padding: 12px 24px; background: #D97706; color: white; border: none; border-radius: 8px; font-size: 16px; cursor: pointer;">
            🖨️ Print or Save as PDF
        </button>
        <button onclick="window.close()" style="padding: 12px 24px; background: #6B7280; color: white; border: none; border-radius: 8px; font-size: 16px; cursor: pointer; margin-left: 10px;">
            Close
        </button>
    </div>
</body>
</html>
    <?php
    return ob_get_clean();
}
