<?php
/**
 * Dashboard Stats API
 * Returns dashboard statistics and recent assessments
 */

require_once '../../includes/config.php';
requireLogin();

header('Content-Type: application/json');

try {
    // Fetch statistics
    $stats = [];
    
    // Total Assessments
    $stmt = $pdo->query("SELECT COUNT(*) FROM assessments WHERE status != 'archived'");
    $totalAssessments = $stmt->fetchColumn();
    $stats[] = [
        'label' => 'Total Assessments',
        'value' => $totalAssessments,
        'icon' => '📋',
        'color' => '#3B82F6',
        'link' => 'assessments.php'
    ];
    
    // High Risk Locations
    $stmt = $pdo->query("SELECT COUNT(*) FROM assessments WHERE overall_risk_level IN ('high', 'critical') AND status = 'completed'");
    $highRisk = $stmt->fetchColumn();
    $stats[] = [
        'label' => 'High Risk',
        'value' => $highRisk,
        'icon' => '⚠️',
        'color' => '#DC2626',
        'link' => 'assessments.php'
    ];
    
    // Pending Actions
    $stmt = $pdo->query("
        SELECT COUNT(DISTINCT ad.id) 
        FROM assessment_details ad 
        JOIN assessments a ON ad.assessment_id = a.id 
        WHERE ad.target_date >= CURDATE() 
        AND a.status = 'completed'
    ");
    $pendingActions = $stmt->fetchColumn();
    $stats[] = [
        'label' => 'Pending Actions',
        'value' => $pendingActions,
        'icon' => '📝',
        'color' => '#F59E0B',
        'link' => 'assessments.php'
    ];
    
    // Total Locations
    $stmt = $pdo->query("SELECT COUNT(*) FROM locations");
    $totalLocations = $stmt->fetchColumn();
    $stats[] = [
        'label' => 'Locations',
        'value' => $totalLocations,
        'icon' => '📍',
        'color' => '#059669',
        'link' => 'locations.php'
    ];
    
    // Recent Assessments
    $stmt = $pdo->prepare("
        SELECT 
            a.id,
            a.assessment_number,
            a.assessment_date,
            a.overall_risk_level,
            a.status,
            l.location_name,
            l.city,
            l.postcode,
            u.full_name as assessor_name
        FROM assessments a
        JOIN locations l ON a.location_id = l.id
        JOIN users u ON a.assessor_id = u.id
        WHERE a.status != 'archived'
        ORDER BY a.created_at DESC
        LIMIT 10
    ");
    $stmt->execute();
    $recentAssessments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Risk Distribution
    $stmt = $pdo->query("
        SELECT 
            overall_risk_level as level,
            COUNT(*) as count
        FROM assessments
        WHERE status = 'completed'
        GROUP BY overall_risk_level
    ");
    $riskDistData = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate percentages
    $total = array_sum(array_column($riskDistData, 'count'));
    $riskDistribution = [];
    foreach ($riskDistData as $risk) {
        $riskDistribution[] = [
            'level' => $risk['level'],
            'count' => (int)$risk['count'],
            'percentage' => $total > 0 ? round(($risk['count'] / $total) * 100, 1) : 0
        ];
    }
    
    // Sort by risk level
    $order = ['critical' => 4, 'high' => 3, 'medium' => 2, 'low' => 1];
    usort($riskDistribution, function($a, $b) use ($order) {
        return ($order[$b['level']] ?? 0) - ($order[$a['level']] ?? 0);
    });
    
    // Return response
    echo json_encode([
        'success' => true,
        'stats' => $stats,
        'recent' => $recentAssessments,
        'risk_distribution' => $riskDistribution,
        'timestamp' => time()
    ]);
    
} catch (PDOException $e) {
    error_log('Dashboard API Error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Failed to fetch dashboard data',
        'message' => 'Please try again later'
    ]);
}
