# Assessment Version Control Guide

## Overview

The version control system tracks all changes to risk assessments, maintaining complete history of revisions, enabling comparison between versions, and supporting archiving of outdated assessments.

## Key Concepts

### Version Numbers
- Each assessment starts at **v1** (original)
- Revisions increment: v1 → v2 → v3...
- Current version is always marked
- Previous versions preserved indefinitely

### Version Types
- **Original** - Initial assessment creation
- **Revision** - Minor updates or corrections
- **Major Revision** - Significant changes requiring new assessment

### Version States
- **Current** - Active version in use
- **Superseded** - Replaced by newer version
- **Archived** - Formally archived with reason
- **Draft** - Work in progress (not yet current)

## Database Structure

### Core Tables

**`assessment_versions`** - Complete version snapshots
```sql
- version_number (1, 2, 3...)
- version_type (original/revision/major_revision)
- revision_reason (why created)
- changes_summary (what changed)
- is_current (boolean)
- supersedes_version (previous version ID)
```

**`assessment_version_details`** - Risk items per version
```sql
- version_id
- risk_item_id
- likelihood, severity, risk_score
- current_controls, additional_actions
```

**`assessment_change_log`** - Audit trail
```sql
- from_version → to_version
- action_type (created/revised/archived/restored)
- change_description
- changed_by, changed_at
```

**`assessment_archives`** - Archive metadata
```sql
- version_id (which version archived)
- archive_reason
- archived_by, archived_at
- can_restore (boolean)
```

## Using Version Control

### View Version History

**Access:** View Assessment → "📚 Version History" button

Shows:
- Visual timeline of all versions
- Version numbers and dates
- Revision reasons
- Current vs superseded status
- Change log entries
- Archive history

### Create a Revision

**Workflow:**
1. **Open Assessment** → Click "📚 Version History"
2. **Click "➕ Create Revision"**
3. **Fill in details:**
   - Revision type (minor/major)
   - Reason for revision (required)
   - Changes summary (optional)
   - Copy existing data or start fresh
4. **Submit** → Creates new version, opens editor

**What Happens:**
- New version created (v2, v3, etc.)
- Old version marked as "superseded"
- Change logged in audit trail
- You're redirected to edit new version

### Compare Versions

**Access:** Version History → "🔄 Compare" button

**Shows:**
- Side-by-side comparison
- Highlighted differences
- Field-by-field changes
- Risk score changes
- Addition/removal of risk items

**Use Cases:**
- Review what changed between versions
- Audit compliance
- Understand risk evolution
- Quality control

### Restore a Version

**Access:** Version History → "↩️ Restore" button (non-current versions)

**Process:**
1. Select version to restore
2. Confirm restoration
3. Creates new revision based on old version
4. Old version data copied to new version
5. New version becomes current

**Important:** Restoration creates a NEW version (doesn't delete current)

## Version Control Workflows

### Regular Updates

**Scenario:** Site conditions changed, need to update assessment

```
Current: v1 (6 months old)
Action: Create revision
Result: v2 (current), v1 (superseded)
Reason: "Updated for new loading bay surface"
```

### Major Re-Assessment

**Scenario:** Significant site changes requiring complete review

```
Current: v3
Action: Create major revision, start fresh
Result: v4 (current), v3 (superseded)
Reason: "Complete re-assessment following warehouse extension"
```

### Correction

**Scenario:** Error in previous assessment needs fixing

```
Current: v2
Action: Create minor revision, copy data
Result: v3 (current), v2 (superseded)
Reason: "Corrected vehicle height restriction measurement"
```

### Seasonal Updates

**Scenario:** Different conditions in winter vs summer

```
Spring: v1 created
Winter: v2 created - "Winter weather conditions assessment"
Spring: v3 created - "Spring conditions update"
```

## Archiving Assessments

### When to Archive

Archive when:
- ✅ Location permanently closed
- ✅ Assessment superseded by site changes
- ✅ Compliance period ended
- ✅ Historical record only
- ✅ Location sold/transferred

Don't archive if:
- ❌ Just needs updating (create revision instead)
- ❌ Temporary closure
- ❌ Seasonal changes

### Archive Process

1. **Version History** → Select version
2. **Archive** → Provide reason
3. **Confirm** → Version marked archived
4. **Logged** → Audit trail updated

**Archived Versions:**
- Remain viewable (read-only)
- Can be restored if needed
- Appear in history with "Archived" badge
- Don't appear in active assessments list

## Change Log & Audit Trail

### Tracked Actions

Every change is logged:
- **Created** - Initial assessment
- **Revised** - New version created
- **Archived** - Version archived
- **Restored** - Old version restored
- **Superseded** - Replaced by newer version

### Log Entry Contains

- Timestamp (precise date/time)
- User who made change
- From version → to version
- Action type
- Description of change

### Viewing Change Log

**Access:** Version History page → "Change Log" section

**Use Cases:**
- Audit compliance
- Track who changed what
- Understand assessment evolution
- Regulatory reporting

## Best Practices

### Revision Reasons

**Good Examples:**
```
✅ "Updated risk scores following installation of new lighting system"
✅ "Re-assessed after customer feedback on delivery access"
✅ "Corrected vehicle type from 'Large Van' to '7.5 Tonne'"
✅ "Annual review - no changes to risk levels"
✅ "Emergency revision following near-miss incident"
```

**Poor Examples:**
```
❌ "Update"
❌ "Changes made"
❌ "V2"
❌ "Fix"
```

### Changes Summary

**Good:**
```
"- Increased pedestrian safety risk from Low to Medium
 - Added new risk item: Overhead power lines
 - Updated current controls for all loading bay risks
 - Changed vehicle type to reflect actual usage"
```

**Poor:**
```
"Updated some things"
"Various changes"
```

### Naming Conventions

**Version Types:**
- Use "revision" for minor updates
- Use "major_revision" only when >50% of assessment changed
- Document reason clearly

### Data Management

**When Creating Revision:**
- Copy data if making incremental changes
- Start fresh if complete re-assessment
- Always provide meaningful revision reason
- Update date to reflect new assessment

## Reports & Analytics

### Version Statistics

Track:
- Total versions per assessment
- Average time between revisions
- Most frequently revised assessments
- Revision reasons breakdown

### Queries

**Find assessments with multiple versions:**
```sql
SELECT assessment_number, version_count
FROM assessments
WHERE version_count > 1
ORDER BY version_count DESC;
```

**View complete history:**
```sql
SELECT av.version_number, av.created_at, 
       u.full_name, av.revision_reason
FROM assessment_versions av
JOIN users u ON av.created_by = u.id
WHERE av.assessment_id = ?
ORDER BY av.version_number;
```

**Compare risk scores between versions:**
```sql
SELECT ri.item_name,
       v1.risk_score as old_score,
       v2.risk_score as new_score,
       (v2.risk_score - v1.risk_score) as change
FROM assessment_version_details v1
JOIN assessment_version_details v2 ON v1.risk_item_id = v2.risk_item_id
JOIN risk_items ri ON v1.risk_item_id = ri.id
WHERE v1.version_id = ? AND v2.version_id = ?
HAVING change != 0;
```

## Integration

### PDF Exports

- PDFs can be generated for any version
- Version number appears in report
- Historical PDFs remain accurate

### Photos

- Photos linked to versions
- Photos visible in version they were added
- Can add photos to new versions

### Custom Fields

- Custom field values preserved per version
- Changes tracked in version history

## Troubleshooting

### Can't Create Revision

**Check:**
- Do you have edit permissions?
- Is assessment in correct status?
- Are required fields filled?

### Missing Version History Button

**Reason:** Assessment only has one version (v1)

**Solution:** Create first revision to enable history

### Restore Not Available

**Reasons:**
- Version is already current
- Version is archived as permanent
- Insufficient permissions

## Migration

### Existing Assessments

All existing assessments automatically:
- Assigned version number 1
- Marked as "original" type
- Set as current version
- Logged in change log

### No Data Loss

Migration is non-destructive:
- All existing data preserved
- Assessments continue working
- Version control is additive

## Summary

**Version Control Enables:**
- ✅ Complete assessment history
- ✅ Track all changes with reasons
- ✅ Compare versions side-by-side
- ✅ Restore previous versions
- ✅ Archive old assessments
- ✅ Audit compliance
- ✅ Quality control

**Best Used For:**
- Site condition changes
- Periodic reviews
- Error corrections
- Regulatory compliance
- Historical tracking

---

**Version control ensures assessment integrity and traceability!** 📚✨
