# Vehicle Types & Delivery Volumes Configuration

## Overview

Vehicle types and delivery volumes are now **fully configurable** through an admin interface instead of being hardcoded. This allows you to customize the dropdown options to match your specific fleet and operations.

## Location in Database

### Tables

**`vehicle_types`** - Stores available vehicle types
```sql
CREATE TABLE vehicle_types (
    id INT PRIMARY KEY,
    type_name VARCHAR(100) UNIQUE,  -- Display name
    description TEXT,                -- Optional description
    display_order INT,               -- Sort order in dropdown
    is_active BOOLEAN,               -- Active/inactive status
    created_at TIMESTAMP
);
```

**`delivery_volumes`** - Stores delivery volume categories
```sql
CREATE TABLE delivery_volumes (
    id INT PRIMARY KEY,
    volume_name VARCHAR(100) UNIQUE,
    description TEXT,
    display_order INT,
    is_active BOOLEAN,
    created_at TIMESTAMP
);
```

### Used In

**`assessments` table:**
- `vehicle_type` VARCHAR(50) - Stores selected vehicle type name
- `delivery_volume` VARCHAR(50) - Stores selected delivery volume name

## Default Values

### Vehicle Types (7 pre-configured)

1. **Small Van** - Small delivery van (e.g., Transit Connect, Caddy)
2. **Large Van** - Large panel van (e.g., Sprinter, Transit)
3. **7.5 Tonne** - 7.5 tonne rigid vehicle
4. **Rigid Truck** - Rigid truck (18-26 tonne)
5. **Articulated Lorry** - Articulated lorry/HGV
6. **Box Truck** - Box truck with tail lift
7. **Flatbed** - Flatbed truck

### Delivery Volumes (6 pre-configured)

1. **Small (1-10 items)** - Small delivery, few items
2. **Medium (11-50 items)** - Medium delivery
3. **Large (51-100 items)** - Large delivery
4. **Very Large (100+ items)** - Very large or full load
5. **Full Pallet** - Full pallet delivery
6. **Multiple Pallets** - Multiple pallets

## Admin Interface

### Accessing the Configuration Page

**Navigation:** Admin Menu → 🚚 Vehicles

**URL:** `/vehicle-config.php`

**Required Role:** Admin only

### Features

#### Two Tabs
- **Vehicle Types** - Manage vehicle options
- **Delivery Volumes** - Manage volume options

#### For Each Item

**View:**
- Name
- Description
- Display order
- Usage count (how many assessments use it)
- Active/inactive status

**Actions:**
- ✏️ Edit - Modify name, description, order, status
- 🗑️ Delete - Remove (only if not used in any assessments)

## Managing Vehicle Types

### Adding a New Vehicle Type

1. **Navigate** to Vehicle Config page
2. **Click** "➕ Add Vehicle Type"
3. **Fill in:**
   - **Vehicle Type Name**: Display name (e.g., "Electric Van")
   - **Description**: Optional notes (e.g., "Electric delivery van for city centers")
   - **Display Order**: Number (0-100, lower = appears first)
4. **Click** "💾 Save"
5. **Result**: New option appears in assessment form dropdown

**Example Use Cases:**
```
Adding specialized vehicles:
- "Refrigerated Van" for cold chain deliveries
- "Curtain Sider" for flexible loading
- "Tail Lift Truck" for heavy items
- "Bicycle Courier" for small urban deliveries
- "Electric Vehicle" for low emission zones
```

### Editing a Vehicle Type

1. **Click** ✏️ Edit button next to vehicle type
2. **Modify** any fields:
   - Change name (updates dropdown immediately)
   - Update description
   - Adjust display order
   - Toggle active/inactive status
3. **Click** "💾 Save"

**Note:** Editing name doesn't affect historical assessments - they keep the old name.

### Deactivating vs Deleting

**Deactivate (Recommended):**
- ✅ Keeps historical data intact
- ✅ Removes from dropdown for new assessments
- ✅ Can be reactivated later
- ✅ Shows usage count

**Delete:**
- ❌ Permanent removal
- ❌ Only allowed if usage count = 0
- ❌ Use only for test entries or mistakes

**How to Deactivate:**
1. Click ✏️ Edit
2. Uncheck "Active" checkbox
3. Save
4. Vehicle type no longer appears in dropdown
5. Historical assessments still show it

### Display Order

Controls the position in dropdown menus:
- **0** = First in list
- **1-99** = Middle positions
- **100** = Last in list

**Best Practice:**
```
0  - Most common vehicle (e.g., "Large Van")
1  - Second most common (e.g., "7.5 Tonne")
2  - Third most common
...
99 - Rarely used vehicles
```

Reorder by editing items and changing their display_order values.

## Managing Delivery Volumes

Works exactly the same as vehicle types:

### Adding New Volume

Example: Add seasonal volumes
```
"Peak Season (200+ items)" - Christmas/holiday rush
"Off-Peak (1-20 items)" - Quiet periods
"Emergency Delivery" - Urgent small deliveries
```

### Common Customizations

**Weight-based:**
```
"Light (< 500kg)"
"Medium (500-1500kg)"
"Heavy (> 1500kg)"
```

**Pallet-based:**
```
"Quarter Pallet"
"Half Pallet"
"Full Pallet"
"2 Pallets"
"3+ Pallets"
```

**Time-based:**
```
"Standard (4-6 hour slot)"
"Express (2 hour slot)"
"Next Day"
"Same Day"
```

## Database Queries

### View All Active Vehicle Types

```sql
SELECT * FROM vehicle_types 
WHERE is_active = 1 
ORDER BY display_order;
```

### Check Usage of Vehicle Type

```sql
SELECT 
    v.type_name,
    COUNT(a.id) as assessment_count
FROM vehicle_types v
LEFT JOIN assessments a ON a.vehicle_type = v.type_name
GROUP BY v.id
ORDER BY assessment_count DESC;
```

### Find Assessments Using Specific Vehicle

```sql
SELECT 
    a.assessment_number,
    a.assessment_date,
    l.location_name
FROM assessments a
JOIN locations l ON a.location_id = l.id
WHERE a.vehicle_type = 'Large Van'
ORDER BY a.assessment_date DESC;
```

### Update Vehicle Type Display Order

```sql
-- Move "Electric Van" to top of list
UPDATE vehicle_types 
SET display_order = 0 
WHERE type_name = 'Electric Van';

-- Renumber others
UPDATE vehicle_types 
SET display_order = display_order + 1 
WHERE type_name != 'Electric Van';
```

### Deactivate Unused Vehicle Types

```sql
-- Find vehicle types with no usage
SELECT v.type_name
FROM vehicle_types v
LEFT JOIN assessments a ON a.vehicle_type = v.type_name
WHERE a.id IS NULL;

-- Deactivate them
UPDATE vehicle_types v
LEFT JOIN assessments a ON a.vehicle_type = v.type_name
SET v.is_active = 0
WHERE a.id IS NULL;
```

## Migration for Existing Systems

### If You Already Have Assessments

**Option 1: Run Migration (Recommended)**

```bash
mysql -u scubatricky_risk -p scubatricky_risk < includes/migration_vehicle_types.sql
```

This creates the tables with default values. Existing assessments keep working because they store the text values directly.

**Option 2: Manual Setup**

```sql
-- Create tables (see includes/schema.sql)
-- Then add your current vehicle types:

INSERT INTO vehicle_types (type_name, display_order) VALUES
('Your Vehicle 1', 1),
('Your Vehicle 2', 2);

INSERT INTO delivery_volumes (volume_name, display_order) VALUES
('Your Volume 1', 1),
('Your Volume 2', 2);
```

### Preserving Historical Data

The system stores the **actual text value** in assessments, not the ID. This means:

✅ **Safe:** Edit/delete vehicle types without affecting old assessments
✅ **Safe:** Rename vehicle types - old assessments keep old names
✅ **Safe:** Deactivate vehicle types - still visible in historical data
❌ **Caution:** Don't manually edit assessment records' vehicle_type field

## Best Practices

### Naming Conventions

**Do:**
- ✅ Use clear, descriptive names
- ✅ Include size/capacity if relevant
- ✅ Use consistent terminology
- ✅ Keep names under 50 characters

**Don't:**
- ❌ Use abbreviations unless standard (e.g., "HGV" is OK)
- ❌ Include punctuation unless necessary
- ❌ Use ALL CAPS
- ❌ Duplicate names

### Organization

**Group Related Types:**
```
Vans:
  - Small Van (order: 10)
  - Large Van (order: 11)
  - Extra Large Van (order: 12)

Trucks:
  - 7.5 Tonne (order: 20)
  - Rigid Truck (order: 21)
  - Articulated (order: 22)
```

Use display_order gaps (10, 20, 30) to allow inserting items later.

### Maintenance

**Regular Review:**
- Monthly: Check usage counts
- Quarterly: Deactivate unused types
- Annually: Review and consolidate

**Don't:**
- Delete types with historical usage
- Change names drastically (better to add new + deactivate old)
- Have too many options (overwhelming for users)

### Recommended Limits

- **Vehicle Types:** 10-15 maximum
- **Delivery Volumes:** 6-10 maximum
- More options = harder to choose = lower data quality

## Troubleshooting

### Vehicle Type Not Appearing in Dropdown

**Possible Causes:**
1. Item is deactivated
2. Display order wrong (very high number)
3. Database not updated after adding

**Check:**
```sql
SELECT * FROM vehicle_types WHERE type_name = 'Your Vehicle';
-- Look at is_active column
```

**Fix:**
```sql
UPDATE vehicle_types 
SET is_active = 1 
WHERE type_name = 'Your Vehicle';
```

### Can't Delete Vehicle Type

**Message:** "Cannot delete: This item is used in X assessment(s)"

**Reason:** Vehicle type is used in historical assessments

**Solution:** Deactivate instead of delete
1. Edit the vehicle type
2. Uncheck "Active"
3. Save

### Dropdown Showing Old Value

**Cause:** Browser cache

**Fix:**
1. Hard refresh: Ctrl+F5 (Windows) or Cmd+Shift+R (Mac)
2. Or clear browser cache

### Database Error on Add/Edit

**Check:**
- Name is not duplicate
- Name length under 100 characters
- Display order is a number

## Integration with Assessments

### How It Works

**Assessment Form:**
1. Page loads → Queries `vehicle_types` and `delivery_volumes`
2. Loops through results → Creates `<option>` tags
3. User selects → Value stored in `assessments.vehicle_type`

**Code (new-assessment.php):**
```php
// Load from database
$vehicle_types = $pdo->query("
    SELECT type_name 
    FROM vehicle_types 
    WHERE is_active = 1 
    ORDER BY display_order
")->fetchAll();

// Generate dropdown
foreach ($vehicle_types as $type) {
    echo "<option value='{$type['type_name']}'>";
    echo $type['type_name'];
    echo "</option>";
}
```

### Validation

Assessment form doesn't validate against available types - this allows historical data to remain valid even if types are later deactivated.

**Example:**
1. 2024: Assessment uses "Delivery Van"
2. 2025: "Delivery Van" deactivated, renamed to "Standard Van"
3. Result: 2024 assessment still shows "Delivery Van" ✅

## API Access (for developers)

### Get Active Vehicle Types

```http
GET /api/get-vehicle-types.php
```

**Response:**
```json
{
  "vehicle_types": [
    {"id": 1, "type_name": "Small Van"},
    {"id": 2, "type_name": "Large Van"}
  ]
}
```

### Get Active Delivery Volumes

```http
GET /api/get-delivery-volumes.php
```

## Summary

**Before:** Vehicle types hardcoded in form
**After:** Fully configurable through admin interface

**Benefits:**
- ✅ Customize to your fleet
- ✅ Add new types anytime
- ✅ Track usage statistics
- ✅ Deactivate without deleting data
- ✅ Control display order
- ✅ No code changes needed

**Admin Access:** Admin menu → 🚚 Vehicles

---

**Vehicle types and delivery volumes are now flexible and maintainable!** 🚚📦
