# SQL Column Error Fix - Unknown column 'l.address'

## Problem

**Error Message:**
```
Error: SQLSTATE[42S22]: Column not found: 1054 Unknown column 'l.address' in 'SELECT'
```

**Cause:**
The PDF export query was trying to select a column called `address` from the `locations` table, but the actual column names are `address_line1` and `address_line2`.

## Database Schema

### Locations Table Columns:

```sql
CREATE TABLE locations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    location_name VARCHAR(200) NOT NULL,
    address_line1 VARCHAR(200) NOT NULL,    -- ✅ Correct
    address_line2 VARCHAR(200),              -- ✅ Correct
    city VARCHAR(100) NOT NULL,
    postcode VARCHAR(10) NOT NULL,
    region VARCHAR(100),
    contact_name VARCHAR(100),
    contact_phone VARCHAR(20),
    contact_email VARCHAR(100),
    -- ... other fields
);
```

**Note:** There is NO `address` column - the address is split into two fields:
- `address_line1` - First line of address (required)
- `address_line2` - Second line of address (optional)

## Solution Implemented

### 1. Fixed SQL Query

**File:** `public/api/export-report.php`

**Before (Incorrect):**
```sql
SELECT a.*, l.location_name, l.address, l.city, l.postcode, 
       l.contact_name, l.contact_phone,
       u.full_name as assessor_name, u.email as assessor_email
FROM assessments a
JOIN locations l ON a.location_id = l.id
JOIN users u ON a.assessor_id = u.id
WHERE a.id = ?
```

**After (Fixed):**
```sql
SELECT a.*, l.location_name, l.address_line1, l.address_line2, l.city, l.postcode, 
       l.contact_name, l.contact_phone,
       u.full_name as assessor_name, u.email as assessor_email
FROM assessments a
JOIN locations l ON a.location_id = l.id
JOIN users u ON a.assessor_id = u.id
WHERE a.id = ?
```

### 2. Fixed HTML Output

**Before (Incorrect):**
```php
<div class="info-label">Address:</div>
<div class="info-value"><?php echo htmlspecialchars($assessment['address']); ?></div>
```

**After (Fixed):**
```php
<div class="info-label">Address:</div>
<div class="info-value">
    <?php 
    echo htmlspecialchars($assessment['address_line1']); 
    if (!empty($assessment['address_line2'])) {
        echo '<br>' . htmlspecialchars($assessment['address_line2']);
    }
    ?>
</div>
```

**Benefits:**
- ✅ Displays both address lines
- ✅ Shows line break between lines
- ✅ Handles missing address_line2 gracefully
- ✅ Proper HTML escaping

## Files Checked

### Files Using Locations Table:

**✅ Fixed:**
- `public/api/export-report.php` - PDF export query

**✅ Already Correct (using l.*):**
- `public/view-assessment.php` - Uses `SELECT l.*`
- `public/assessments.php` - Uses `SELECT l.*`
- `public/dashboard.php` - Uses `SELECT l.*`
- `public/compare-versions.php` - Uses `SELECT l.*`
- `public/create-revision.php` - Uses `SELECT l.*`
- `public/version-history.php` - Uses `SELECT l.*`

**Why `l.*` Works:**
When you use `SELECT l.*`, it selects ALL columns from the locations table, including `address_line1` and `address_line2`, so these queries work correctly.

**Why Explicit Column Names Failed:**
The export-report.php explicitly listed `l.address` which doesn't exist.

## Verification Queries

### Check Column Names:

```sql
-- Show all columns in locations table
DESCRIBE locations;

-- Or
SHOW COLUMNS FROM locations;

-- Should show:
-- address_line1 | varchar(200) | NO
-- address_line2 | varchar(200) | YES
-- (NO 'address' column)
```

### Test Address Data:

```sql
-- Check address data
SELECT id, location_name, address_line1, address_line2, city, postcode
FROM locations
LIMIT 5;
```

### Find Queries Using Wrong Column:

```bash
# Search all PHP files for 'l.address' (not followed by _)
grep -rn "l\.address[^_]" /path/to/public/ --include="*.php"

# Should return empty after fix
```

## Address Display Patterns

### Correct Ways to Display Address:

**Option 1: Concatenate (inline)**
```php
<?php 
$full_address = $location['address_line1'];
if (!empty($location['address_line2'])) {
    $full_address .= ', ' . $location['address_line2'];
}
echo htmlspecialchars($full_address);
?>
```

**Option 2: Multi-line (recommended for reports)**
```php
<?php 
echo htmlspecialchars($location['address_line1']); 
if (!empty($location['address_line2'])) {
    echo '<br>' . htmlspecialchars($location['address_line2']);
}
?>
```

**Option 3: Function (reusable)**
```php
function formatAddress($location) {
    $address = htmlspecialchars($location['address_line1']);
    if (!empty($location['address_line2'])) {
        $address .= '<br>' . htmlspecialchars($location['address_line2']);
    }
    return $address;
}

// Usage:
echo formatAddress($location);
```

## Common Mistakes to Avoid

### ❌ Don't Do This:
```sql
-- Wrong: Column doesn't exist
SELECT l.address FROM locations l

-- Wrong: Trying to concatenate in SQL without checking null
SELECT CONCAT(l.address_line1, ', ', l.address_line2) as full_address

-- Wrong: Not handling null address_line2
echo $location['address_line1'] . '<br>' . $location['address_line2']; // Shows <br>null
```

### ✅ Do This:
```sql
-- Correct: Use actual column names
SELECT l.address_line1, l.address_line2 FROM locations l

-- Correct: Handle null in SQL
SELECT CONCAT_WS(', ', l.address_line1, NULLIF(l.address_line2, '')) as full_address

-- Correct: Check in PHP
if (!empty($location['address_line2'])) {
    echo '<br>' . htmlspecialchars($location['address_line2']);
}
```

## Testing PDF Export

### Test Steps:

1. **Navigate to Assessment**
   ```
   Login → Assessments → View any assessment
   ```

2. **Generate PDF**
   ```
   Click "📄 Generate PDF"
   Click "Generate Report"
   ```

3. **Verify Success**
   ```
   Should show success message
   No SQL errors
   ```

4. **Check Report**
   ```
   Click "Print / Save as PDF"
   Report opens in new window
   Address section shows correctly:
   - Line 1 of address
   - Line 2 of address (if exists)
   - City, Postcode
   ```

### Test Query Directly:

```bash
# Test the fixed query
mysql -u scubatricky_risk -p scubatricky_risk -e "
SELECT a.id, l.location_name, l.address_line1, l.address_line2, l.city, l.postcode
FROM assessments a
JOIN locations l ON a.location_id = l.id
LIMIT 5;
"

# Should work without errors
```

## Optional: Add Virtual Column

If you want to add `address` as a virtual column for backward compatibility:

```sql
-- Add virtual column that concatenates address lines
ALTER TABLE locations 
ADD COLUMN address VARCHAR(400) 
GENERATED ALWAYS AS (
    CASE 
        WHEN address_line2 IS NULL OR address_line2 = '' 
        THEN address_line1
        ELSE CONCAT(address_line1, ', ', address_line2)
    END
) VIRTUAL;
```

**Pros:**
- ✅ Provides `address` column
- ✅ Automatically updated
- ✅ No storage space (virtual)
- ✅ Backward compatible

**Cons:**
- ❌ Not necessary (queries fixed)
- ❌ Extra complexity
- ❌ Not recommended

**Recommendation:** Don't add it - keep queries using actual columns.

## Error Prevention

### Code Review Checklist:

When writing queries against locations table:

- [ ] Use `address_line1` and `address_line2`, NOT `address`
- [ ] Handle `address_line2` null/empty values
- [ ] Use proper HTML escaping
- [ ] Test with locations that have no address_line2
- [ ] Test with locations that have both lines

### Example Good Query:

```php
// ✅ Good - Explicit columns, handles null
$stmt = $pdo->prepare("
    SELECT l.location_name, l.address_line1, l.address_line2, l.city, l.postcode
    FROM locations l
    WHERE l.id = ?
");
$stmt->execute([$location_id]);
$location = $stmt->fetch();

// Display
echo htmlspecialchars($location['address_line1']);
if (!empty($location['address_line2'])) {
    echo '<br>' . htmlspecialchars($location['address_line2']);
}
```

## Summary

**Problem:** SQL query trying to select non-existent `address` column

**Root Cause:** Locations table uses `address_line1` and `address_line2`, not `address`

**Fix:**
1. ✅ Updated SQL query to use correct columns
2. ✅ Updated HTML output to display both lines
3. ✅ Added null checking for optional line 2
4. ✅ Verified all other files use correct columns

**Testing:**
- ✅ PDF export now works
- ✅ Address displays correctly
- ✅ No more SQL errors

**Prevention:**
- Always check actual database schema
- Use `DESCRIBE table_name` to verify columns
- Test queries before deploying
- Handle nullable columns properly

**The SQL column error is now completely fixed!** ✅
