# Risk Categories & Items Management Guide

## Overview

Risk categories organize the assessment process into logical groups. Each category contains multiple risk items that assessors evaluate during site visits. This system is fully configurable through the admin interface.

## Quick Access

**Navigation:** Admin Panel → ⚠️ Risk Categories  
**Or:** Admin Menu → Admin Panel → Risk Categories card

## Database Structure

### Tables

**`risk_categories`** - Main category groupings
```sql
CREATE TABLE risk_categories (
    id INT PRIMARY KEY,
    category_name VARCHAR(100),      -- Display name
    description TEXT,                -- Optional description
    display_order INT,               -- Sort order
    is_active BOOLEAN               -- Active/inactive status
);
```

**`risk_items`** - Individual risk items within categories
```sql
CREATE TABLE risk_items (
    id INT PRIMARY KEY,
    category_id INT,                 -- Links to category
    item_name VARCHAR(200),          -- Risk description
    description TEXT,                -- Additional guidance
    display_order INT,               -- Sort within category
    is_active BOOLEAN               -- Active/inactive status
);
```

**`assessment_details`** - Assessment evaluations
```sql
CREATE TABLE assessment_details (
    id INT PRIMARY KEY,
    assessment_id INT,               -- Links to assessment
    risk_item_id INT,                -- Which risk item
    likelihood INT,                  -- 1-5 rating
    severity INT,                    -- 1-5 rating
    risk_score INT,                  -- Calculated: likelihood × severity
    current_controls TEXT,           -- Existing controls
    additional_actions TEXT,         -- Required actions
    action_owner VARCHAR(100),       -- Responsible person
    target_date DATE                 -- Completion date
);
```

## Default Categories

The system comes with 6 pre-configured categories:

1. **Vehicle Access** - Risks related to vehicle access and maneuverability
2. **Loading Bay** - Risks in the loading/unloading area
3. **Pedestrian Safety** - Risks to pedestrians and delivery personnel
4. **Environmental Hazards** - Weather, lighting, and environmental conditions
5. **Manual Handling** - Risks during manual handling of goods
6. **Security** - Security-related risks

Each category contains multiple risk items (30+ total default items).

## Managing Categories

### Viewing Categories

1. Go to **Admin Panel → Risk Categories**
2. See list of all categories with:
   - Category name
   - Description
   - Display order
   - Number of items in category
   - Active/inactive status
   - Action buttons

### Adding a Category

1. Click **"➕ Add Category"** button
2. Fill in details:
   - **Category Name**: Short, descriptive (e.g., "Equipment Safety")
   - **Description**: What this category covers
   - **Display Order**: Position in assessments (0 = first)
3. Click **"💾 Save Category"**

**Example Categories:**
```
"Equipment Safety" - Risks from equipment and tools
"Site Conditions" - Physical site condition risks
"Traffic Management" - Vehicle and pedestrian traffic
"Working at Height" - Elevated work platform risks
"Emergency Procedures" - Emergency access and procedures
```

### Editing a Category

1. Click **✏️ Edit** button next to category
2. Modify any fields:
   - Change name (updates in assessments going forward)
   - Update description
   - Adjust display order
   - Toggle active/inactive
3. Click **"💾 Save Category"**

**Important:** Editing category name doesn't affect existing assessments - they retain the name at the time of assessment.

### Deactivating a Category

**Why Deactivate?**
- Temporarily remove from assessments
- Keep all historical data intact
- Can be reactivated later
- Better than deleting

**How to Deactivate:**
1. Click ✏️ Edit
2. Uncheck "Active" checkbox
3. Save
4. Category hidden from new assessments
5. Existing assessments still show it

### Deleting a Category

**Restrictions:**
- Can only delete if no risk items in category
- Must delete all items first, or
- Move items to another category

**Steps:**
1. Ensure category has 0 items
2. Click **🗑️ Delete** button
3. Confirm deletion
4. Category permanently removed

## Managing Risk Items

### Accessing Risk Items

1. Go to Risk Categories page
2. Click **"📋 Items"** button next to category
3. See all risk items in that category

### Adding a Risk Item

1. Click **"➕ Add Risk Item"**
2. Fill in details:
   - **Risk Item Name**: Specific risk description
   - **Description**: Guidance for assessors (optional)
   - **Display Order**: Position within category
3. Click **"💾 Save Risk Item"**

**Good Risk Item Names:**
```
✅ "Uneven loading surface"
✅ "Poor lighting in bay area"
✅ "Narrow access requiring reversing"
✅ "Overhead power lines present"
✅ "Slippery surface when wet"

❌ "Surface problems"  (too vague)
❌ "Issue 1"           (not descriptive)
❌ "Check the bay"     (not specific)
```

### Risk Item Examples by Category

**Vehicle Access:**
- Narrow entrance requiring tight turns
- Steep gradient at entry/exit
- Limited turning circle in yard
- Overhead height restriction
- Width restriction on approach

**Loading Bay:**
- Uneven loading surface
- No dock leveler available
- Poor lighting in bay area
- Obstructions in loading zone
- Distance from delivery point

**Pedestrian Safety:**
- No designated walkways
- Pedestrian/vehicle route overlap
- Poor visibility at intersections
- Uneven walking surfaces
- Inadequate lighting for pedestrians

**Environmental Hazards:**
- Exposure to adverse weather
- Poor drainage causing puddles
- Slippery surface when wet
- Inadequate lighting during winter
- Exposed to strong winds

**Manual Handling:**
- Long carrying distance
- Steps/stairs to access point
- No handling equipment available
- Awkward access requiring twisting
- Heavy items without mechanical aid

**Security:**
- Site access control measures
- Lone working considerations
- Public access to delivery area
- Valuable goods exposure risk
- CCTV coverage of area

### Editing a Risk Item

1. Click **✏️ Edit** next to item
2. Modify:
   - Item name
   - Description
   - Display order
   - Active/inactive status
3. Save changes

### Display Order Strategy

**Category Level:**
```
0  - Vehicle Access (first evaluated)
1  - Loading Bay (second)
2  - Pedestrian Safety
3  - Environmental Hazards
4  - Manual Handling
5  - Security (last)
```

**Item Level (within category):**
```
0  - Most critical/common risk first
1  - Second most important
2  - Third...
...
99 - Least common/lowest priority
```

**Tip:** Leave gaps (0, 10, 20, 30) to allow inserting items later.

### Deactivating vs Deleting Items

**Deactivate When:**
- Risk no longer relevant to current operations
- Seasonal risk (reactivate when needed)
- Want to keep historical data
- Might need again in future

**Delete When:**
- Item was added by mistake
- Duplicate of another item
- Never used in any assessments (usage count = 0)

## Assessment Workflow

### How Categories & Items are Used

**During Assessment Creation:**

1. **Assessor opens "New Assessment"**
2. **System loads all active categories** (ordered by display_order)
3. **For each category:**
   - Shows category name and description
   - Lists all active risk items in that category
   - Assessor rates each item:
     - Likelihood (1-5)
     - Severity (1-5)
     - Risk Score calculated (likelihood × severity)
     - Current controls (text)
     - Additional actions needed (text)
     - Action owner (person)
     - Target date (deadline)

4. **Results stored** in assessment_details table
5. **Overall risk level** calculated from all items

### Risk Matrix

**5×5 Risk Matrix:**
```
           Severity →
         1    2    3    4    5
L   1    1    2    3    4    5
i   2    2    4    6    8   10
k   3    3    6    9   12   15
e   4    4    8   12   16   20
l   5    5   10   15   20   25
↓
```

**Risk Levels:**
- **1-4**: Low (Green)
- **5-9**: Medium (Yellow)
- **10-16**: High (Orange)
- **17-25**: Critical (Red)

## Usage Statistics

### View Item Usage

The system tracks how many assessments have evaluated each risk item.

**In Risk Items Page:**
- Usage count shown for each item
- Helps identify:
  - Most commonly evaluated risks
  - Rarely used items (candidates for removal)
  - Popular risk patterns

**Query Usage:**
```sql
SELECT 
    ri.item_name,
    rc.category_name,
    COUNT(ad.id) as times_assessed
FROM risk_items ri
JOIN risk_categories rc ON ri.category_id = rc.id
LEFT JOIN assessment_details ad ON ri.id = ad.risk_item_id
GROUP BY ri.id
ORDER BY times_assessed DESC;
```

### Reports

**Most Common Risks:**
```sql
SELECT 
    ri.item_name,
    AVG(ad.risk_score) as avg_risk_score,
    COUNT(*) as assessment_count
FROM assessment_details ad
JOIN risk_items ri ON ad.risk_item_id = ri.id
GROUP BY ri.id
HAVING COUNT(*) > 10
ORDER BY avg_risk_score DESC;
```

## Best Practices

### Category Organization

**Do:**
- ✅ Group related risks logically
- ✅ Keep categories broad but distinct
- ✅ Use 5-10 categories total
- ✅ Order by assessment workflow
- ✅ Write clear descriptions

**Don't:**
- ❌ Create too many categories (overwhelming)
- ❌ Overlap categories (confusing)
- ❌ Use vague category names
- ❌ Mix different risk types in one category

### Risk Item Writing

**Be Specific:**
```
Good: "Loading bay surface uneven with 3-5cm height differences"
Poor: "Bay surface issues"

Good: "Narrow access lane (2.4m width) requires precise maneuvering"
Poor: "Access problems"

Good: "Overhead power lines at 4.5m height pose strike risk for tall vehicles"
Poor: "Overhead hazard"
```

**Include Context:**
```
Item Name: "Distance from bay to storage area"
Description: "Assess carrying distance for manual handling. >20m may require trolley/pallet truck."

Item Name: "Lighting quality in loading area"
Description: "Check illumination levels, especially relevant for early morning/evening deliveries."
```

### Maintenance Schedule

**Monthly:**
- Review usage statistics
- Identify unused items
- Check for duplicate items
- Update descriptions based on feedback

**Quarterly:**
- Deactivate obsolete items
- Add new items based on incidents
- Reorder items by frequency
- Update category descriptions

**Annually:**
- Complete review of all categories
- Consolidate similar items
- Update to reflect operational changes
- Archive old assessments if needed

## Troubleshooting

### Category Won't Delete

**Error:** "Category has X items"

**Solution:**
1. Go to that category's items page
2. Either:
   - Delete all items (if unused), or
   - Move items to another category (manually), or
   - Deactivate category instead

### Item Won't Delete

**Error:** "Used in X assessments"

**Solution:**
- Don't delete - deactivate instead
- This preserves historical data
- Item won't appear in new assessments
- Old assessments still show it

### Items Not Appearing in Assessments

**Check:**
1. Is item marked as active?
2. Is parent category active?
3. Clear browser cache (Ctrl+F5)
4. Check display_order not too high (999+)

**Fix:**
```sql
-- Check status
SELECT item_name, is_active FROM risk_items WHERE id = X;

-- Reactivate if needed
UPDATE risk_items SET is_active = 1 WHERE id = X;
```

### Wrong Display Order

**Problem:** Items appearing in wrong order

**Solution:**
1. Edit each item
2. Set display_order values:
   - 0, 10, 20, 30... (allows easy reordering)
3. Save each item
4. Refresh assessment page

## Advanced Usage

### Copying Categories Between Systems

**Export:**
```sql
-- Export category and items
SELECT * FROM risk_categories WHERE id = X;
SELECT * FROM risk_items WHERE category_id = X;
```

**Import:**
```sql
-- Import to new system
INSERT INTO risk_categories (category_name, description, display_order) 
VALUES ('Category Name', 'Description', 0);

-- Get new category_id, then:
INSERT INTO risk_items (category_id, item_name, description, display_order)
VALUES (new_category_id, 'Item Name', 'Description', 0);
```

### Bulk Updates

**Deactivate Multiple Items:**
```sql
UPDATE risk_items 
SET is_active = 0 
WHERE category_id = X AND item_name LIKE '%keyword%';
```

**Renumber Display Order:**
```sql
-- For category_id = 3, renumber items by 10s
SET @row_number = 0;
UPDATE risk_items 
SET display_order = (@row_number:=@row_number + 10) 
WHERE category_id = 3 
ORDER BY display_order, item_name;
```

## Summary

**Key Points:**
- Categories organize risk items into logical groups
- Items are specific risks assessed at each location
- Both categories and items can be added/edited/deactivated
- Deactivate > Delete (preserves historical data)
- Usage statistics help optimize your risk list
- Display order controls assessment workflow

**Quick Links:**
- Admin Panel → Risk Categories
- Categories → Items (click 📋 Items button)
- Add/Edit/Delete via modal forms
- View usage statistics in table

---

**Fully customizable risk assessment framework!** ⚠️✨
