# Risk Assessment System for Supermarket Delivery Points

A comprehensive, self-hosted web application for conducting and managing risk assessments at supermarket delivery points across the UK.

## 🎯 Features

### Core Functionality
- **User Authentication** - Secure login system with role-based access (Admin, Assessor, Viewer)
- **Risk Assessment Management** - Create, edit, view, and archive detailed risk assessments
- **Enhanced Location Management** - Comprehensive delivery point information including:
  - Full address details with postcode
  - GPS coordinates with interactive map placement
  - What3Words integration for precise location
  - Embedded Google Maps with satellite view
  - Opening hours (configurable per day)
  - Delivery time windows (multiple slots supported)
  - Contact information
  - Photo attachments
- **Risk Matrix System** - Visual 5x5 risk matrix with likelihood and severity scoring
- **Customizable Risk Categories** - Pre-configured categories including:
  - Vehicle Access
  - Loading Bay Safety
  - Pedestrian Safety
  - Environmental Hazards
  - Manual Handling
  - Security

### Advanced Features
- **Offline Support** - Full PWA (Progressive Web App) with IndexedDB for offline data storage
- **Auto-save** - Automatic draft saving to prevent data loss
- **PDF Report Generation** - Professional, customizable PDF reports with company branding
- **Photo Upload** - Attach photos to risk assessments (with offline storage)
- **Audit Trail** - Complete logging of all user actions
- **Dashboard Analytics** - Visual statistics and risk distribution charts
- **Responsive Design** - Modern, mobile-friendly interface with industrial safety theme

## 🛠️ Technology Stack

- **Backend**: PHP 7.4+ with PDO
- **Database**: MySQL 5.7+ / MariaDB 10.3+
- **Frontend**: HTML5, CSS3, JavaScript (Vanilla)
- **PDF Generation**: Python 3 with ReportLab
- **Offline Storage**: IndexedDB, Service Workers
- **PWA**: Manifest, Service Worker for offline capabilities

## 📋 Requirements

### Server Requirements
- PHP 7.4 or higher
- MySQL 5.7+ or MariaDB 10.3+
- Python 3.7+ (for PDF generation)
- Web server (Apache/Nginx)
- SSL certificate (recommended for PWA features)

### PHP Extensions Required
- PDO
- pdo_mysql
- json
- mbstring
- session

### Python Packages Required
```bash
pip install reportlab --break-system-packages
```

## 🚀 Installation

### 1. Clone or Download Files

Place all files in your web server's document root or a subdirectory:

```bash
/var/www/html/risk-system/
├── public/              # Web accessible files
│   ├── css/
│   ├── js/
│   ├── api/
│   ├── images/
│   ├── *.php
│   └── manifest.json
├── includes/            # Configuration and shared code
│   ├── config.php
│   └── schema.sql
├── templates/           # Page templates
│   └── header.php
└── reports/            # Generated PDF reports (needs write permission)
```

### 2. Database Setup

Create the database and import the schema:

```bash
# Login to MySQL
mysql -u scubatricky_risk -p

# Create database and user
CREATE DATABASE scubatricky_risk CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'riskassess_user'@'localhost' IDENTIFIED BY 'your_secure_password';
GRANT ALL PRIVILEGES ON scubatricky_risk.* TO 'riskassess_user'@'localhost';
FLUSH PRIVILEGES;

# Import schema
mysql -u scubatricky_risk -p scubatricky_risk < includes/schema.sql
```

### 3. Configure Database Connection

Edit `includes/config.php`:

```php
define('DB_HOST', 'localhost');
define('DB_USER', 'riskassess_user');
define('DB_PASS', 'your_secure_password');
define('DB_NAME', 'scubatricky_risk');
define('SITE_URL', 'https://your-domain.com/risk-system');
```

### 4. Set File Permissions

```bash
# Make reports directory writable
chmod 755 reports/
chown www-data:www-data reports/

# Make Python script executable
chmod +x public/api/generate_pdf.py
```

### 5. Configure API Keys

The application uses external APIs for enhanced location features:

#### Google Maps API (Required for map features)
1. Go to [Google Cloud Console](https://console.cloud.google.com/)
2. Create a new project or select existing
3. Enable these APIs:
   - Maps JavaScript API
   - Geocoding API
   - Places API
4. Create credentials (API Key)
5. Restrict the API key to your domain
6. Replace `YOUR_GOOGLE_MAPS_API_KEY` in:
   - `public/add-location.php`
   - `public/edit-location.php`
   - `public/view-location.php`

```php
// Find and replace this line in the files above:
<script src="https://maps.googleapis.com/maps/api/js?key=YOUR_GOOGLE_MAPS_API_KEY&libraries=places"></script>
```

#### What3Words API (Optional but recommended)
1. Sign up at [what3words.com/developers](https://developer.what3words.com/)
2. Get your API key
3. Add implementation in `public/add-location.php` fetchWhat3Words function:

```javascript
const response = await fetch(
    `https://api.what3words.com/v3/convert-to-3wa?coordinates=${lat},${lng}&key=YOUR_W3W_API_KEY`
);
```

**Note**: The application works without these API keys, but location features will be limited. Manual entry of coordinates and What3Words is still supported.

### 6. Apache Configuration (if using Apache)

Create or edit `.htaccess` in the public directory:

```apache
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteBase /risk-system/
    
    # Redirect to HTTPS
    RewriteCond %{HTTPS} off
    RewriteRule ^(.*)$ https://%{HTTP_HOST%{REQUEST_URI} [L,R=301]
    
    # Protect includes directory
    RewriteRule ^includes/ - [F,L]
</IfModule>

# Security headers
<IfModule mod_headers.c>
    Header set X-Content-Type-Options "nosniff"
    Header set X-Frame-Options "SAMEORIGIN"
    Header set X-XSS-Protection "1; mode=block"
</IfModule>
```

### 6. Nginx Configuration (if using Nginx)

Add to your server block:

```nginx
location /risk-system/ {
    try_files $uri $uri/ /risk-system/index.php?$query_string;
    
    # Security
    add_header X-Content-Type-Options "nosniff";
    add_header X-Frame-Options "SAMEORIGIN";
    add_header X-XSS-Protection "1; mode=block";
}

location ~ ^/risk-system/includes/ {
    deny all;
    return 404;
}

location ~ \.php$ {
    fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
    fastcgi_index index.php;
    include fastcgi_params;
}
```

## 👤 Default Login Credentials

**Username**: `admin`  
**Password**: `Admin123!`

⚠️ **Important**: Change the default password immediately after first login!

## 📱 PWA Installation

The app can be installed as a Progressive Web App:

1. Open the app in Chrome/Edge on desktop or mobile
2. Look for the "Install" button in the address bar
3. Click to install as a standalone app
4. Access from your home screen/app drawer

## 🎨 Customization

### Branding

Edit `public/css/style.css` to change colors:

```css
:root {
    --primary: #D97706;        /* Primary brand color */
    --primary-dark: #B45309;   /* Darker shade */
    --primary-light: #FCD34D;  /* Lighter shade */
    /* ... */
}
```

### Risk Categories

Add or modify risk categories in the database:

```sql
INSERT INTO risk_categories (category_name, description, display_order) 
VALUES ('Your Category', 'Description', 7);

INSERT INTO risk_items (category_id, item_name, description, display_order) 
VALUES (7, 'Risk Item Name', 'Description', 1);
```

### PDF Report Template

Modify `public/api/generate_pdf.py` to customize:
- Company logo and branding
- Color scheme
- Section layout
- Additional fields

## 📊 User Roles

### Admin
- Full system access
- User management
- View all assessments
- System configuration

### Assessor
- Create assessments
- Edit own assessments
- View all assessments
- Generate reports

### Viewer
- View assessments
- Generate reports
- No edit permissions

## 🔧 Troubleshooting

### PDF Generation Fails

1. Check Python is installed: `python3 --version`
2. Install ReportLab: `pip3 install reportlab --break-system-packages`
3. Check script permissions: `chmod +x public/api/generate_pdf.py`
4. Check reports directory is writable: `chmod 755 reports/`

### Database Connection Errors

1. Verify MySQL is running: `systemctl status mysql`
2. Check credentials in `includes/config.php`
3. Ensure database user has correct permissions

### Offline Mode Not Working

1. Ensure you're using HTTPS (required for Service Workers)
2. Check browser console for Service Worker registration errors
3. Clear browser cache and re-register Service Worker

### Session Issues

1. Check PHP session configuration: `php -i | grep session`
2. Ensure session directory is writable
3. Increase session timeout if needed in `config.php`

## 🔒 Security Recommendations

1. **Use HTTPS** - Essential for PWA and security
2. **Change Default Password** - Immediately after installation
3. **Regular Backups** - Automate database backups
4. **Update Regularly** - Keep PHP, MySQL, and dependencies updated
5. **Restrict File Permissions** - Minimum necessary permissions
6. **Use Strong Passwords** - Enforce password policies
7. **Enable Audit Logging** - Monitor user actions

## 📈 Maintenance

### Database Backup

```bash
# Daily backup
mysqldump -u riskassess_user -p scubatricky_risk > backup_$(date +%Y%m%d).sql

# Restore from backup
mysql -u riskassess_user -p scubatricky_risk < backup_20260106.sql
```

### Clear Old Audit Logs

```sql
-- Delete audit logs older than 90 days
DELETE FROM audit_log WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY);
```

### Archive Old Assessments

```sql
-- Archive assessments older than 2 years
UPDATE assessments 
SET status = 'archived' 
WHERE assessment_date < DATE_SUB(CURDATE(), INTERVAL 2 YEAR) 
AND status != 'archived';
```

## 📄 License

This application is provided as-is for self-hosted use. Modify as needed for your organization.

## 🤝 Support

For issues or questions:
1. Check the troubleshooting section
2. Review server logs: `/var/log/apache2/error.log` or `/var/log/nginx/error.log`
3. Check PHP error log
4. Review browser console for JavaScript errors

## 📝 Changelog

### Version 1.0.0 (2026-01-06)
- Initial release
- Core assessment functionality
- Offline support
- PDF generation
- User management
- Audit logging

---

**Built with ❤️ for supermarket delivery safety**
