# PDF Generation - Complete Fix & Troubleshooting

## Current Status

The PDF generation system uses **PHP-based HTML reports** that can be printed/saved as PDF using the browser.

## How PDF Generation Works

### User Flow:
```
1. User clicks "📄 Generate PDF" on assessment page
2. Modal opens
3. User clicks "Generate Report" button
4. JavaScript calls: api/export-report.php?assessment_id=X
5. Server generates HTML report
6. Returns success with two URLs:
   - print_url: Opens report in new window
   - download_url: Downloads HTML file
7. User clicks "Print / Save as PDF"
8. Browser print dialog opens
9. User saves as PDF
```

### Technical Flow:
```
view-assessment.php
    → generatePDF() JavaScript function
        → fetch('api/export-report.php?assessment_id=X')
            → Generates HTML
            → Saves to reports/ directory
            → Returns JSON response
        → Displays success with buttons
            → Print / Save as PDF button
            → Download HTML button
```

## Files Involved

### Main Files:
1. **view-assessment.php** - Page with Generate PDF button
2. **api/export-report.php** - Generates HTML report
3. **api/print-report.php** - Opens report for printing
4. **api/download-report.php** - Downloads HTML file

### Directory:
- **reports/** - Where generated reports are saved

## Common Issues & Fixes

### Issue 1: Button Doesn't Respond

**Symptoms:**
- Click "Generate PDF" button
- Nothing happens
- No modal or error

**Causes:**
- JavaScript error
- Missing modal HTML

**Fix:**
```bash
# Check browser console for errors (F12 → Console)

# Verify modal exists in view-assessment.php
grep "pdf-export-modal" public/view-assessment.php

# Verify JavaScript function exists
grep "function generatePDF" public/view-assessment.php
```

### Issue 2: "File Not Found" or 404 Error

**Symptoms:**
- Button works but shows error
- Console shows 404 for api/export-report.php

**Causes:**
- API file missing
- Wrong file path

**Fix:**
```bash
# Check if file exists
ls -la public/api/export-report.php

# Check file permissions
chmod 644 public/api/export-report.php

# Verify reports directory
mkdir -p reports
chmod 755 reports
chown www-data:www-data reports
```

### Issue 3: Reports Directory Error

**Symptoms:**
- Error: "Failed to create reports directory"
- Error: "Permission denied"

**Causes:**
- No write permissions
- Directory doesn't exist

**Fix:**
```bash
# Create directory
mkdir -p reports

# Set permissions
chmod 755 reports

# Set ownership (web server user)
chown www-data:www-data reports

# Or if using different user
chown your-web-user:your-web-group reports
```

### Issue 4: Database Connection Error

**Symptoms:**
- Error: "Assessment not found"
- Error: "Database error"

**Causes:**
- Wrong database credentials
- Database not accessible

**Fix:**
```bash
# Check database config
cat includes/config.php | grep DB_

# Test database connection
mysql -u scubatricky_risk -p scubatricky_risk -e "SELECT COUNT(*) FROM assessments;"

# Check if assessment exists
mysql -u scubatricky_risk -p scubatricky_risk -e "SELECT id, assessment_number FROM assessments LIMIT 5;"
```

### Issue 5: Blank or Empty Report

**Symptoms:**
- Report generates but is empty
- Missing data in report

**Causes:**
- Assessment has no data
- SQL query failing
- Data not properly escaped

**Fix:**
```php
// Add debugging to export-report.php
error_log("Assessment data: " . print_r($assessment, true));
error_log("Details count: " . count($details));

// Check PHP error log
tail -f /var/log/apache2/error.log
// or
tail -f /var/log/php-fpm/error.log
```

### Issue 6: Cannot Save as PDF

**Symptoms:**
- Report opens but can't save as PDF
- Print dialog doesn't show "Save as PDF"

**Causes:**
- Browser doesn't support PDF saving
- Popup blocker active

**Fix:**
- **Chrome/Edge:** Ctrl+P → Destination: "Save as PDF"
- **Firefox:** Ctrl+P → Destination: "Save to PDF"  
- **Safari:** Cmd+P → PDF dropdown → "Save as PDF"
- Disable popup blocker for your domain
- Or right-click link → "Open in new tab"

## Testing PDF Generation

### Step-by-Step Test:

1. **Navigate to Assessment**
   ```
   Login → Assessments → View any assessment
   ```

2. **Click Generate PDF**
   ```
   Should see modal open
   ```

3. **Check Browser Console**
   ```
   Press F12 → Console tab
   Should see no errors
   ```

4. **Click Generate Report**
   ```
   Should see "Generating..." message
   Wait 1-2 seconds
   Should see success message with buttons
   ```

5. **Check Network Tab**
   ```
   F12 → Network tab
   Look for: export-report.php
   Status should be: 200 OK
   Response should be: {"success":true,...}
   ```

6. **Click Print / Save as PDF**
   ```
   New window should open
   Should see formatted report
   ```

7. **Print or Save**
   ```
   Ctrl+P (or Cmd+P on Mac)
   Choose "Save as PDF"
   Select location and filename
   Click Save
   ```

### Manual Test of API:

```bash
# Test the API directly
curl -b cookies.txt "http://yoursite.com/api/export-report.php?assessment_id=1"

# Should return JSON like:
# {"success":true,"filename":"assessment_A001_...html","download_url":"...","print_url":"..."}
```

## Debugging Steps

### Enable PHP Error Logging:

```php
// Add to top of export-report.php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
```

### Add Debug Output:

```php
// In export-report.php, after line 22:
error_log("Assessment ID: " . $assessment_id);
error_log("Assessment found: " . ($assessment ? 'Yes' : 'No'));
error_log("Details count: " . count($details));
error_log("Output path: " . $output_path);
```

### Check Generated Files:

```bash
# List generated reports
ls -lah reports/

# View a report
cat reports/assessment_XXX.html

# Check file size (should be > 1KB)
du -h reports/assessment_XXX.html
```

## Browser Testing

Test in multiple browsers:

### Chrome/Edge:
```
- Open DevTools (F12)
- Console tab: Check for errors
- Network tab: Check API calls
- Click Generate PDF
- Verify modal works
- Verify API returns 200
- Verify new window opens
- Test Ctrl+P → Save as PDF
```

### Firefox:
```
- Same steps as Chrome
- Test print function
- Verify PDF saving works
```

### Safari:
```
- Develop menu → Show JavaScript Console
- Test same flow
- Use Cmd+P for printing
```

## File Permissions

Correct permissions:

```bash
# PHP files
chmod 644 public/api/*.php

# Reports directory
chmod 755 reports
chown www-data:www-data reports

# Config file
chmod 640 includes/config.php

# Upload directory (for photos)
chmod 755 uploads
chown www-data:www-data uploads
```

## Server Requirements

### Required:
- ✅ PHP 7.4 or higher
- ✅ MySQL 5.7 or higher
- ✅ PDO extension enabled
- ✅ Write access to reports/ directory
- ✅ Session support

### Not Required:
- ❌ Python (not needed anymore)
- ❌ ReportLab library
- ❌ Shell exec permissions
- ❌ Special PDF libraries

## Verifying the Fix

### Checklist:

1. ✅ **File exists:** `public/api/export-report.php`
2. ✅ **File exists:** `public/api/print-report.php`
3. ✅ **File exists:** `public/api/download-report.php`
4. ✅ **Directory exists:** `reports/`
5. ✅ **Permissions correct:** `755` on reports/
6. ✅ **Button exists:** "Generate PDF" in view-assessment.php
7. ✅ **Modal exists:** pdf-export-modal in view-assessment.php
8. ✅ **JavaScript exists:** generatePDF() function
9. ✅ **Database accessible:** Connection works
10. ✅ **No PHP errors:** Check error logs

### Quick Test Script:

```php
<?php
// Save as test-pdf.php in public/ directory
require_once '../includes/config.php';

echo "<h1>PDF Generation Test</h1>";

// Test 1: Database connection
echo "<h2>1. Database Connection</h2>";
try {
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM assessments");
    $count = $stmt->fetch()['count'];
    echo "✅ Connected. Found {$count} assessments.<br>";
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>";
}

// Test 2: Reports directory
echo "<h2>2. Reports Directory</h2>";
$reports_dir = __DIR__ . '/../reports';
if (file_exists($reports_dir)) {
    if (is_writable($reports_dir)) {
        echo "✅ Reports directory exists and is writable.<br>";
    } else {
        echo "❌ Reports directory exists but is NOT writable.<br>";
    }
} else {
    echo "❌ Reports directory does NOT exist.<br>";
}

// Test 3: API files
echo "<h2>3. API Files</h2>";
$files = ['export-report.php', 'print-report.php', 'download-report.php'];
foreach ($files as $file) {
    $path = __DIR__ . '/api/' . $file;
    if (file_exists($path)) {
        echo "✅ {$file} exists<br>";
    } else {
        echo "❌ {$file} MISSING<br>";
    }
}

// Test 4: Generate sample report
echo "<h2>4. Sample Report Generation</h2>";
try {
    $stmt = $pdo->query("SELECT id FROM assessments LIMIT 1");
    $assessment = $stmt->fetch();
    if ($assessment) {
        $test_url = "api/export-report.php?assessment_id=" . $assessment['id'];
        echo "✅ Test URL: <a href='{$test_url}' target='_blank'>{$test_url}</a><br>";
        echo "Click link to test PDF generation.<br>";
    } else {
        echo "⚠️ No assessments found in database.<br>";
    }
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "<br>";
}

echo "<hr><p><strong>Delete this file after testing!</strong></p>";
?>
```

## Alternative Solutions

If still not working:

### Option 1: Direct PHP PDF
Use TCPDF library (requires composer):
```bash
composer require tecnickcom/tcpdf
```

### Option 2: Use wkhtmltopdf
System-level PDF converter:
```bash
apt-get install wkhtmltopdf
```

### Option 3: Cloud PDF Service
Use services like:
- PDFShift
- DocRaptor  
- HTML2PDF API

## Summary

**Current Implementation:**
- ✅ Pure PHP (no dependencies)
- ✅ Generates HTML reports
- ✅ Uses browser to save as PDF
- ✅ Works on any server
- ✅ No external libraries needed

**Advantages:**
- Simple & reliable
- No installation needed
- High-quality output
- Easy to customize
- Works everywhere

**To Fix Issues:**
1. Check browser console for errors
2. Verify API files exist
3. Check reports/ directory permissions
4. Test database connection
5. Review server error logs
6. Use test script above

**Still Not Working?**
1. Check STYLESHEET_REVERSION.md status
2. Verify JavaScript is loaded
3. Check for conflicting code
4. Review browser network tab
5. Enable PHP error logging
6. Test API endpoint directly

---

**The PDF generation system is designed to be simple and reliable. Follow the troubleshooting steps above to resolve any issues.** 📄✅
