# Login Issues - Troubleshooting Guide

## ⚠️ Admin Login Not Working?

If you're having trouble logging in as admin, follow these steps:

## Quick Fix (Recommended)

### Method 1: Use Emergency Reset Tool

1. **Navigate to the reset page in your browser:**
   ```
   https://your-domain.com/reset-admin.php
   ```

2. **Set a new password** (or use default "Admin123!")

3. **Click "Reset Admin Password"**

4. **Login with new credentials**

5. **IMPORTANT: Delete reset-admin.php immediately after use!**
   ```bash
   rm /path/to/public/reset-admin.php
   ```

### Method 2: Command Line Reset

If you have SSH access:

```bash
cd /path/to/risk-assessment-app/includes
php reset-password.php
# Choose option 1 or 2
```

### Method 3: Direct Database Update

If you have MySQL access:

```sql
-- Connect to MySQL
mysql -u scubatricky_risk -p scubatricky_risk

-- Generate a new hash for password "Admin123!"
-- Then update the admin user
UPDATE users 
SET password_hash = '$2y$10$rQZ9kN.6YXxL0V8BKXKj4eRZKJxJ0gxKvX8cHJGqK8qzN6JqP6/J2' 
WHERE username = 'admin';

-- Verify the update
SELECT username, email, role FROM users WHERE username = 'admin';
```

## Common Issues & Solutions

### Issue 1: "Invalid username or password"

**Causes:**
- Wrong password
- Password hash corrupted
- Admin user doesn't exist
- Caps Lock is on

**Solutions:**
1. Check Caps Lock is OFF
2. Try: `admin` / `Admin123!` (case-sensitive)
3. Use reset-admin.php to create new password
4. Check if admin user exists:
   ```sql
   SELECT * FROM users WHERE username = 'admin';
   ```

### Issue 2: "Database connection failed"

**Causes:**
- Wrong database credentials
- MySQL not running
- Database doesn't exist

**Solutions:**
1. Check `includes/config.php` has correct credentials
2. Verify MySQL is running:
   ```bash
   systemctl status mysql
   ```
3. Check database exists:
   ```sql
   SHOW DATABASES LIKE 'scubatricky_risk';
   ```

### Issue 3: User shows as inactive

**Check and fix:**
```sql
-- Check if user is active
SELECT username, is_active FROM users WHERE username = 'admin';

-- Activate if needed
UPDATE users SET is_active = 1 WHERE username = 'admin';
```

### Issue 4: Session errors

**Clear sessions:**
```bash
# Clear PHP session files
rm -rf /tmp/sess_*
# Or wherever your session.save_path points to
```

**Check session configuration in `includes/config.php`:**
```php
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
session_start();
```

## Password Requirements

The default password is: **Admin123!**

- Username: `admin` (lowercase)
- Password: `Admin123!` (case-sensitive)
  - Capital A
  - Capital M
  - Numbers: 123
  - Exclamation mark: !

## Creating Admin Password Hash Manually

If you need to generate a hash for a different password:

### Using PHP CLI:
```bash
php -r "echo password_hash('YourPassword123!', PASSWORD_DEFAULT) . PHP_EOL;"
```

### Using the web interface:
1. Create a temporary file: `hash.php`
```php
<?php
$password = 'YourPassword123!';
echo password_hash($password, PASSWORD_DEFAULT);
// Delete this file after use!
?>
```
2. Visit: `https://your-domain.com/hash.php`
3. Copy the hash
4. Delete the file immediately!

### Using reset-password.php script:
```bash
cd includes
php reset-password.php
# Choose option 3 to generate hash only
```

## Verifying Password Hash

Test if a hash matches a password:

```php
<?php
$password = 'Admin123!';
$hash = '$2y$10$rQZ9kN.6YXxL0V8BKXKj4eRZKJxJ0gxKvX8cHJGqK8qzN6JqP6/J2';

if (password_verify($password, $hash)) {
    echo "✓ Password matches!";
} else {
    echo "✗ Password does not match!";
}
?>
```

## Security Best Practices

1. **Change default password immediately** after first login
2. **Delete reset-admin.php** after using it
3. **Delete any test hash.php files**
4. **Use strong passwords** (12+ characters)
5. **Restrict database access** to localhost only
6. **Enable HTTPS** for production
7. **Regular backups** of database

## Database Schema Check

Verify the users table exists and has correct structure:

```sql
DESCRIBE users;

-- Should show:
-- id, username, email, password_hash, full_name, role, 
-- created_at, last_login, is_active
```

If table is missing or corrupted:
```bash
mysql -u scubatricky_risk -p scubatricky_risk < includes/schema.sql
```

## Creating Additional Admin Users

Via database:
```sql
INSERT INTO users (username, email, password_hash, full_name, role) VALUES
('newadmin', 'newadmin@example.com', '$2y$10$yourHashHere', 'New Admin', 'admin');
```

Via reset script (after logging in):
1. Navigate to Users page
2. Click "Add User"
3. Set role to "Admin"

## Debugging Login Issues

Enable error display temporarily in `includes/config.php`:

```php
// Add at the top for debugging ONLY
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// REMOVE BEFORE GOING TO PRODUCTION!
```

Check PHP error log:
```bash
tail -f /var/log/php_errors.log
# or
tail -f /var/log/apache2/error.log
```

## Fresh Installation

If all else fails, reinstall:

```bash
# 1. Backup any data you want to keep
mysqldump -u scubatricky_risk -p scubatricky_risk > backup.sql

# 2. Drop and recreate database
mysql -u scubatricky_risk -p -e "DROP DATABASE IF EXISTS scubatricky_risk;"
mysql -u scubatricky_risk -p -e "CREATE DATABASE scubatricky_risk CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"

# 3. Import fresh schema
mysql -u scubatricky_risk -p scubatricky_risk < includes/schema.sql

# 4. Verify admin user exists
mysql -u scubatricky_risk -p scubatricky_risk -e "SELECT username, email, role FROM users WHERE username = 'admin';"

# 5. Test login
```

## Testing Login Flow

Create a test script `test-login.php`:

```php
<?php
require_once '../includes/config.php';

$username = 'admin';
$password = 'Admin123!';

try {
    $stmt = $pdo->prepare("SELECT id, username, password_hash, full_name, role, is_active FROM users WHERE username = ?");
    $stmt->execute([$username]);
    $user = $stmt->fetch();
    
    if (!$user) {
        echo "❌ User not found!\n";
        exit;
    }
    
    echo "✓ User found: {$user['username']}\n";
    echo "  Email: {$user['email']}\n";
    echo "  Role: {$user['role']}\n";
    echo "  Active: " . ($user['is_active'] ? 'Yes' : 'No') . "\n";
    echo "  Hash: {$user['password_hash']}\n\n";
    
    if (password_verify($password, $user['password_hash'])) {
        echo "✓ Password is CORRECT!\n";
        echo "\nLogin should work with:\n";
        echo "  Username: admin\n";
        echo "  Password: Admin123!\n";
    } else {
        echo "❌ Password is INCORRECT!\n";
        echo "Use reset-admin.php to fix.\n";
    }
    
} catch (PDOException $e) {
    echo "Database error: " . $e->getMessage() . "\n";
}

// DELETE THIS FILE AFTER TESTING!
?>
```

Run: `php test-login.php` or visit in browser, then DELETE it!

## Contact Support

If none of these solutions work:

1. Check server error logs
2. Verify PHP version (7.4+)
3. Verify MySQL version (5.7+)
4. Check file permissions
5. Verify web server configuration

## Quick Reference

**Default Credentials:**
- Username: `admin`
- Password: `Admin123!`

**Reset Methods:**
1. Web: `/reset-admin.php`
2. CLI: `php includes/reset-password.php`
3. SQL: `UPDATE users SET password_hash = '...' WHERE username = 'admin';`

**Files to Delete After Use:**
- `reset-admin.php`
- `test-login.php`
- `hash.php`
- Any other temporary test files

---

**Remember: Security first! Always delete temporary password reset tools after use.**
