# Location Management Guide

## Enhanced Location Features

The Risk Assessment System includes comprehensive location management with advanced mapping and scheduling features.

## 📍 Location Information

### Basic Details
- **Location Name**: Unique identifier for the delivery point
- **Full Address**: Complete UK address with postcode
- **Contact Information**: On-site contact person with phone and email
- **Additional Notes**: Free-text field for special instructions

### 🗺️ Mapping Features

#### GPS Coordinates
Precise latitude and longitude for exact delivery point location.

**How to Set:**
1. **Auto-detect from Address**: Enter address and click "Find on Map"
2. **Click on Map**: Simply click the delivery point on the interactive map
3. **Use Current Location**: Click "Use Current Location" if at the site
4. **Manual Entry**: Enter coordinates directly (format: 51.507400, -0.127800)

**Benefits:**
- Accurate navigation for drivers
- Distance calculations
- Route optimization
- Integration with GPS devices

#### What3Words Integration
What3Words divides the world into 3m x 3m squares, each with a unique three-word address.

**Example**: `///filled.count.soap`

**How to Use:**
1. Enter GPS coordinates first (automatically fetches What3Words if API is configured)
2. Or manually enter from [what3words.com](https://what3words.com)
3. Look up any location on What3Words website and copy the three words

**Benefits:**
- Easy to communicate over phone/radio
- No confusion with similar street names
- Precise to 3 meters
- Works in remote areas without formal addresses

**UK Examples:**
- `///index.home.raft` - Buckingham Palace entrance
- `///sober.poets.medals` - 10 Downing Street
- `///piano.wash.icons` - Tesco Extra, Manchester

#### Google Maps Integration

**Interactive Map Features:**
- **Satellite View**: See actual delivery point from above
- **Street View**: Ground-level view of approach routes
- **Hybrid Mode**: Satellite with road labels
- **Click to Set**: Click anywhere to update delivery point
- **Draggable Marker**: Fine-tune exact position

**Navigation Tools:**
- **Get Directions**: Opens Google Maps with route planning
- **Copy Coordinates**: Quick clipboard copy for GPS devices
- **Open in Google Maps**: Full Google Maps interface
- **Generate Maps URL**: Shareable link to exact location

### 🕐 Opening Hours

Set operational hours for each day of the week.

**Configuration:**
- Individual times for each day (Monday-Sunday)
- Opening and closing times
- Leave blank for closed days
- 24-hour format support

**Use Cases:**
- Schedule assessments during open hours
- Plan deliveries around operating times
- Identify after-hours access requirements
- Coordinate with site staff availability

**Example Configuration:**
```
Monday:    06:00 - 22:00
Tuesday:   06:00 - 22:00
Wednesday: 06:00 - 22:00
Thursday:  06:00 - 23:00  (Late night shopping)
Friday:    06:00 - 23:00
Saturday:  07:00 - 22:00
Sunday:    10:00 - 16:00  (Reduced hours)
```

### 🚚 Delivery Windows

Define specific time slots when deliveries are permitted or preferred.

**Multiple Windows Supported:**
You can add unlimited delivery windows to accommodate different schedules.

**Configuration Fields:**
- **Start Time**: Beginning of delivery window
- **End Time**: End of delivery window
- **Days**: Which days this window applies (e.g., "Mon-Fri", "Weekends", "Daily")

**Example Scenarios:**

**Standard Schedule:**
```
Window 1: 06:00 - 10:00 (Mon-Fri)  - Morning deliveries
Window 2: 14:00 - 18:00 (Mon-Fri)  - Afternoon deliveries
Window 3: 07:00 - 12:00 (Sat-Sun)  - Weekend deliveries
```

**Peak Restrictions:**
```
Window 1: 06:00 - 09:00 (Daily)    - Early morning (avoid rush)
Window 2: 14:00 - 16:00 (Daily)    - Mid-afternoon (avoid lunch rush)
Window 3: 20:00 - 22:00 (Mon-Sat)  - Evening (after peak shopping)
```

**Seasonal Variations:**
```
Window 1: 05:00 - 08:00 (Mon-Fri)  - Christmas/Holiday early slot
Window 2: 22:00 - 01:00 (Thu-Sat)  - Late night restocking
```

## 🎯 Workflow Examples

### Adding a New Location

1. **Navigate**: Go to "Locations" → "Add Location"

2. **Enter Basic Info**:
   ```
   Location Name: Tesco Extra Manchester Arndale
   Address Line 1: 49 High Street
   City: Manchester
   Postcode: M4 3AH
   ```

3. **Set Coordinates**:
   - Click "Find on Map" to auto-detect
   - Or drag the map marker to exact delivery bay
   - Coordinates auto-populate: 53.483959, -2.239182

4. **Add What3Words**:
   - Auto-filled if API configured
   - Or manually enter: `sober.poets.medals`

5. **Configure Hours**:
   ```
   Mon-Sat: 06:00 - 23:00
   Sunday:  10:00 - 18:00
   ```

6. **Set Delivery Windows**:
   ```
   06:00-09:00 (Mon-Sat)     - Morning slot
   14:00-17:00 (Mon-Sat)     - Afternoon slot
   10:00-14:00 (Sunday)      - Sunday only
   ```

7. **Add Notes**:
   ```
   Delivery bay accessed from rear via Brook Street.
   Height restriction: 4.2m
   Security code required: #1234
   Contact site manager 30 mins before arrival.
   ```

8. **Save**: Click "Create Location"

### Updating Location Details

1. From any page showing the location, click "Edit Location"
2. Modify any fields
3. Use map to adjust coordinates if delivery point moved
4. Update delivery windows for seasonal changes
5. Click "Update Location"

### Viewing Location on Map

1. Go to "Locations" → Select location → "View"
2. Interactive map shows:
   - Satellite view of delivery point
   - Street view option
   - Surrounding area
   - Access routes
3. Click "Get Directions" for turn-by-turn navigation
4. Click "Street View" to see ground-level view

## 🔧 API Configuration

### Google Maps API Setup

**Step 1: Create Project**
1. Go to [Google Cloud Console](https://console.cloud.google.com/)
2. Create new project: "Risk Assessment Maps"
3. Enable billing (free tier available)

**Step 2: Enable APIs**
Enable these three APIs:
- Maps JavaScript API (for displaying maps)
- Geocoding API (for address → coordinates)
- Places API (for autocomplete and place details)

**Step 3: Create API Key**
1. Go to "Credentials"
2. Create credentials → API Key
3. Copy the key

**Step 4: Restrict API Key** (Important for security!)
1. Click "Edit API key"
2. Application restrictions: HTTP referrers
3. Add your website URLs:
   ```
   https://yourdomain.com/*
   https://www.yourdomain.com/*
   ```
4. API restrictions: Restrict key
5. Select only the 3 APIs you enabled
6. Save

**Step 5: Add to Application**
Edit these files and replace `YOUR_GOOGLE_MAPS_API_KEY`:
- `public/add-location.php`
- `public/edit-location.php`
- `public/view-location.php`

Find this line:
```javascript
<script src="https://maps.googleapis.com/maps/api/js?key=YOUR_GOOGLE_MAPS_API_KEY&libraries=places"></script>
```

Replace with:
```javascript
<script src="https://maps.googleapis.com/maps/api/js?key=AIzaSyBxxxxxxxxxxxxxxxxxxxxxxxxxxxxx&libraries=places"></script>
```

**Cost**: Free tier includes:
- 28,000 map loads per month
- $200 monthly credit
- More than sufficient for typical usage

### What3Words API Setup

**Step 1: Register**
1. Go to [developer.what3words.com](https://developer.what3words.com/)
2. Sign up for free account
3. Create API key

**Step 2: Add to Application**
Edit `public/add-location.php`, find the `fetchWhat3Words` function:

```javascript
async function fetchWhat3Words(lat, lng) {
    try {
        const response = await fetch(
            `https://api.what3words.com/v3/convert-to-3wa?coordinates=${lat},${lng}&key=YOUR_W3W_API_KEY`
        );
        const data = await response.json();
        document.getElementById('what3words').value = data.words;
    } catch (error) {
        console.error('What3Words error:', error);
    }
}
```

Replace `YOUR_W3W_API_KEY` with your actual key.

**Cost**: Free tier includes:
- 25,000 requests per month
- More than sufficient for location management

## 💡 Best Practices

### GPS Coordinates
- ✅ Use satellite view to identify exact delivery bay
- ✅ Place marker at vehicle unloading point, not store entrance
- ✅ Verify coordinates by checking Street View
- ❌ Don't place marker on main building if delivery is at rear

### What3Words
- ✅ Share with drivers for easy verbal navigation
- ✅ Use for locations without clear street addresses
- ✅ Verify words by visiting what3words.com link
- ❌ Don't confuse with similar-sounding words

### Opening Hours
- ✅ Update for seasonal changes (Christmas hours, etc.)
- ✅ Note any 24-hour days separately
- ✅ Include public holiday variations in notes
- ❌ Don't forget to update after permanent changes

### Delivery Windows
- ✅ Coordinate with site management
- ✅ Consider traffic patterns
- ✅ Account for peak shopping times
- ✅ Add separate windows for different days if needed
- ❌ Don't overlap with major congestion periods

### Notes Field
Include:
- Special access instructions
- Height/weight restrictions
- Security requirements
- Contact protocols
- Parking arrangements
- Loading equipment available

## 📊 Database Schema

New location fields in the database:

```sql
opening_hours        JSON     -- Day-by-day hours
delivery_windows     JSON     -- Array of delivery slots
latitude             DECIMAL  -- GPS latitude (-90 to 90)
longitude            DECIMAL  -- GPS longitude (-180 to 180)
what3words          VARCHAR  -- Three word address
google_maps_place_id VARCHAR  -- Google Place ID
google_maps_url     TEXT     -- Direct Maps link
```

## 🔍 Troubleshooting

### Maps Not Loading
- Check API key is correct
- Verify APIs are enabled in Google Console
- Check browser console for errors
- Ensure HTTP referrers are configured correctly

### Coordinates Not Auto-Filling
- Verify Geocoding API is enabled
- Check address is complete and valid
- Try clicking on map manually
- Check API key restrictions

### What3Words Not Working
- Verify API key is added to code
- Check monthly quota not exceeded
- Test API key at developer.what3words.com
- Manual entry still works without API

### Map Shows Wrong Location
- Check latitude/longitude format (decimal degrees)
- Verify coordinates are for UK (lat: 49-61, lng: -8 to 2)
- Use "Find on Map" to recalculate
- Check no minus signs were omitted

## 🚀 Future Enhancements

Potential additions to location features:
- Route optimization for multiple deliveries
- Traffic pattern analysis
- Delivery time estimation
- Driver feedback integration
- Photo uploads of delivery bays
- 3D building models
- Drone aerial photography
- Integration with vehicle telemetry

---

**The enhanced location features ensure accurate navigation, optimal scheduling, and comprehensive delivery point information for safer, more efficient operations.**
