# Form Submission Errors - Troubleshooting Guide

## Error: "Error submitting form. Please try again."

This error occurs when creating a new assessment. Here are the solutions:

### ✅ Solution 1: Update JavaScript (FIXED)

The issue was that JavaScript was intercepting ALL form submissions and trying to send them as JSON, but the PHP expects normal form data.

**The fix is already included in the latest archive.**

The updated `js/app.js` now allows the assessment form to submit normally to PHP.

### ✅ Solution 2: Ensure Locations Exist

Assessments require at least one location. If no locations exist:

1. Go to **Locations** menu
2. Click **"Add Location"**
3. Create at least one delivery location
4. Return to **New Assessment**

The form now shows a warning if no locations exist.

### ✅ Solution 3: Check Browser Console

Open browser developer tools (F12) and check for errors:

**Chrome/Edge:**
1. Press F12
2. Click "Console" tab
3. Try to submit the form
4. Look for red error messages

**Common errors and fixes:**

#### Error: "Failed to fetch"
**Cause:** Network issue or incorrect form action URL
**Fix:** Check that the form is submitting to the correct URL

#### Error: "SyntaxError: Unexpected token"
**Cause:** Server returning HTML instead of JSON
**Fix:** This is expected with the new code - PHP returns HTML, not JSON

#### Error: "TypeError: Cannot read property"
**Cause:** Missing form fields
**Fix:** Ensure you've selected a location and filled required fields

### ✅ Solution 4: Check PHP Errors

Enable error display temporarily:

Edit `includes/config.php` and add at the top:
```php
<?php
// TEMPORARY - Remove after debugging!
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
```

Then try to submit the form and look for PHP errors on the page.

**Don't forget to remove these lines after debugging!**

### ✅ Solution 5: Check Database Connection

The error might be from database issues:

```php
<?php
// Test database connection
require_once '../includes/config.php';

try {
    $stmt = $pdo->query("SELECT COUNT(*) FROM locations");
    echo "✓ Database connected. Locations: " . $stmt->fetchColumn();
    
    $stmt = $pdo->query("SELECT COUNT(*) FROM risk_categories");
    echo "<br>✓ Risk categories: " . $stmt->fetchColumn();
    
    $stmt = $pdo->query("SELECT COUNT(*) FROM risk_items");
    echo "<br>✓ Risk items: " . $stmt->fetchColumn();
    
} catch (PDOException $e) {
    echo "✗ Database error: " . $e->getMessage();
}
?>
```

Save as `test-db.php` in public folder, visit it, then delete it.

### ✅ Solution 6: Check Required Fields

The assessment form requires:
- ✅ Assessment Number (auto-generated)
- ✅ Location (dropdown selection)
- ✅ Assessment Date
- ✅ Assessment Time
- ✅ Overall Risk Level (dropdown)

Make sure all these are filled before submitting.

### ✅ Solution 7: Verify File Permissions

Ensure the database and session directories are writable:

```bash
# Check write permissions
ls -la /var/lib/php/sessions/
ls -la /var/run/mysqld/

# Fix if needed
sudo chown www-data:www-data /var/lib/php/sessions/
sudo chmod 755 /var/lib/php/sessions/
```

### ✅ Solution 8: Check Apache/Nginx Logs

```bash
# Apache
tail -f /var/log/apache2/error.log

# Nginx
tail -f /var/log/nginx/error.log

# Look for PHP errors or warnings while submitting form
```

## Testing the Fix

After updating to the latest archive:

1. **Clear browser cache:**
   - Chrome: Ctrl+Shift+Delete → Clear cached files
   - Or hard refresh: Ctrl+F5

2. **Test form submission:**
   - Go to "New Assessment"
   - Fill in all required fields
   - Select a location
   - Click "Save Assessment"
   - Should redirect to view page (not show error)

3. **Verify in database:**
   ```sql
   SELECT * FROM assessments ORDER BY created_at DESC LIMIT 1;
   -- Should show your new assessment
   ```

## Understanding the Fix

### Before (Broken):
```javascript
// JavaScript was intercepting ALL forms
async handleFormSubmit(event) {
    event.preventDefault();  // ← This broke normal submission!
    // Try to send as JSON (PHP expects form data)
}
```

### After (Fixed):
```javascript
async handleFormSubmit(event) {
    // Check if this is assessment form
    if (form.id === 'assessment-form') {
        return;  // Let it submit normally to PHP
    }
    event.preventDefault();  // Only prevent for other forms
}
```

## Alternative: Disable JavaScript for Testing

If you want to test without JavaScript:

1. Open browser DevTools (F12)
2. Go to Settings (gear icon)
3. Check "Disable JavaScript"
4. Refresh page
5. Try submitting form
6. Should work (proves it's a JS issue)

Don't forget to re-enable JavaScript after testing!

## Still Not Working?

### Create a Minimal Test Form

Create `test-form.php` in public folder:

```php
<?php
require_once '../includes/config.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    echo "<h2>Form Data Received:</h2>";
    echo "<pre>";
    print_r($_POST);
    echo "</pre>";
    exit;
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>Test Form</title>
</head>
<body>
    <h1>Test Form Submission</h1>
    <form method="POST">
        <input type="text" name="test_field" value="test value" required>
        <button type="submit">Submit</button>
    </form>
</body>
</html>
```

Visit it and submit. If this works, the issue is specific to the assessment form. If this fails, there's a deeper PHP/server issue.

## Common Causes Summary

1. ✅ **JavaScript intercepting form** (FIXED in latest version)
2. ⚠️ **No locations in database** (add at least one location)
3. ⚠️ **Missing required fields** (check all required fields are filled)
4. ⚠️ **Database connection issue** (check config.php credentials)
5. ⚠️ **PHP errors** (check error logs)
6. ⚠️ **Browser cache** (hard refresh with Ctrl+F5)
7. ⚠️ **Session issues** (check session permissions)

## Quick Checklist

- [ ] Downloaded latest archive (with JavaScript fix)
- [ ] Cleared browser cache (Ctrl+F5)
- [ ] Created at least one location
- [ ] Filled all required fields
- [ ] Checked browser console for errors
- [ ] Checked PHP error log
- [ ] Verified database connection works
- [ ] Tested with JavaScript disabled (should work)

## Contact Support

If still having issues:
1. Open browser console (F12)
2. Try to submit form
3. Screenshot any errors
4. Check server error logs
5. Provide error details for further help

---

**The fix is already in the latest archive download. Just clear your browser cache and it should work!**
