# System Customization Guide

## Overview

The System Customization feature allows administrators to personalize the Risk Assessment System with custom colors, logo, and branding to match their organization's identity.

## Access

**Location:** Admin Panel → System Customization

**Requirements:**
- Admin role required
- Database table `system_settings` must exist

## Features

### 1. System Name Customization
- Change the system name displayed in:
  - Header/logo
  - Browser title
  - PDF reports
  - Email notifications

### 2. Logo Upload
- Upload custom organization logo
- Replaces default ⚠️ icon
- Supported formats: PNG, JPG, SVG, WebP
- Maximum size: 2MB
- Recommended dimensions: 200x50px

### 3. Color Scheme Customization
- Customize 8 color values:
  - Primary (main brand color)
  - Primary Dark (hover states)
  - Primary Light (backgrounds)
  - Secondary (secondary buttons)
  - Success (green, low risk)
  - Warning (orange, medium risk)
  - Danger (red, high risk)
  - Info (blue, info messages)

### 4. Preset Color Schemes
- 6 ready-to-use color schemes:
  - Default (Orange)
  - Blue
  - Green
  - Purple
  - Red
  - Teal

## Installation

### Step 1: Run Migration

```bash
# If upgrading existing installation
mysql -u scubatricky_risk -p scubatricky_risk < includes/migration_system_settings.sql

# If new installation, system_settings table is already in complete_schema.sql
```

### Step 2: Create Uploads Directory

```bash
# Create branding directory
mkdir -p public/uploads/branding
chmod 755 public/uploads/branding
chown www-data:www-data public/uploads/branding
```

### Step 3: Test Access

```
Navigate to: Admin Panel → System Customization
```

## Usage

### Changing System Name

1. Go to System Customization
2. Find "System Name" section
3. Enter new name (e.g., "ABC Company Risk Management")
4. Click "Save System Name"
5. Refresh any page to see changes

### Uploading Logo

1. Go to System Customization
2. Find "System Logo" section
3. Click "Choose File"
4. Select your logo image
5. Click "Upload Logo"
6. Logo appears in header immediately

**Logo Guidelines:**
- Use transparent background (PNG/SVG)
- Keep it simple and readable
- Recommended width: 150-200px
- Recommended height: 30-50px
- Test on dark background

### Customizing Colors

#### Manual Color Selection:

1. Go to System Customization
2. Find "Color Scheme" section
3. Click color picker for each color
4. Adjust to desired color
5. See live preview below
6. Click "Save Color Scheme"
7. Refresh page to apply

#### Using Preset Schemes:

1. Scroll to "Preset Color Schemes"
2. Click desired preset
3. Colors auto-fill in form
4. Preview updates instantly
5. Click "Save Color Scheme"
6. Refresh to apply

### Resetting to Default

1. Scroll to "Color Scheme" section
2. Click "Reset to Default" button
3. Confirm action
4. Default orange theme restored

## Technical Details

### Database Storage

**Table:** `system_settings`

```sql
CREATE TABLE system_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) NOT NULL UNIQUE,
    setting_value TEXT,
    setting_type ENUM('string', 'json', 'boolean', 'integer'),
    description TEXT,
    updated_by INT,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);
```

**Settings Stored:**

| Key | Type | Description |
|-----|------|-------------|
| `system_name` | string | System name |
| `logo_filename` | string | Logo filename |
| `color_scheme` | json | Custom colors |
| `timezone` | string | System timezone |
| `date_format` | string | Date format |
| `time_format` | string | Time format |

### File Storage

**Logo Files:**
- Location: `public/uploads/branding/`
- Naming: `logo.{extension}`
- Only one logo file at a time
- Old logo deleted on new upload

### CSS Generation

When colors are saved:

1. Colors stored in database as JSON
2. Custom CSS file generated: `public/css/custom-theme.css`
3. File contains CSS variables:

```css
:root {
    --primary: #3B82F6;
    --primary-dark: #2563EB;
    --primary-light: #93C5FD;
    /* ... etc */
}
```

4. Header loads custom-theme.css if exists
5. Overrides default colors in style.css

### Header Integration

**Updated `templates/header.php`:**

```php
// Get custom settings
$customLogo = getCustomLogo($pdo);
$systemName = getSystemName($pdo);

// Display custom logo if exists
<?php if ($customLogo): ?>
    <img src="<?php echo $customLogo; ?>" alt="Logo">
<?php else: ?>
    <div class="logo-icon">⚠️</div>
    <span><?php echo $systemName; ?></span>
<?php endif; ?>

// Load custom theme CSS
<?php if (file_exists('css/custom-theme.css')): ?>
    <link rel="stylesheet" href="css/custom-theme.css">
<?php endif; ?>
```

## Color Scheme Examples

### Default (Orange)
```css
--primary: #D97706
--primary-dark: #B45309
--primary-light: #FCD34D
```
**Use Case:** Safety/warning focus, industrial theme

### Blue
```css
--primary: #3B82F6
--primary-dark: #2563EB
--primary-light: #93C5FD
```
**Use Case:** Professional, corporate, trust

### Green
```css
--primary: #10B981
--primary-dark: #059669
--primary-light: #6EE7B7
```
**Use Case:** Environmental, positive, growth

### Purple
```css
--primary: #8B5CF6
--primary-dark: #7C3AED
--primary-light: #C4B5FD
```
**Use Case:** Creative, modern, premium

### Red
```css
--primary: #EF4444
--primary-dark: #DC2626
--primary-light: #FCA5A5
```
**Use Case:** Emergency, urgent, critical focus

### Teal
```css
--primary: #14B8A6
--primary-dark: #0D9488
--primary-light: #5EEAD4
```
**Use Case:** Healthcare, wellness, calm

## Troubleshooting

### Issue: Logo Not Appearing

**Causes:**
- File not uploaded properly
- Wrong permissions
- File path incorrect

**Fix:**
```bash
# Check file exists
ls -la public/uploads/branding/

# Fix permissions
chmod 755 public/uploads/branding
chmod 644 public/uploads/branding/*

# Check database
mysql -u scubatricky_risk -p scubatricky_risk -e \
  "SELECT * FROM system_settings WHERE setting_key = 'logo_filename';"
```

### Issue: Colors Not Applying

**Causes:**
- CSS file not generated
- Browser cache
- CSS file not loaded

**Fix:**
```bash
# Check CSS file exists
ls -la public/css/custom-theme.css

# View contents
cat public/css/custom-theme.css

# Should contain :root { ... }

# Clear browser cache: Ctrl+F5 or Cmd+Shift+R
```

### Issue: Custom Theme Not Loading

**Causes:**
- File permissions
- Wrong file path
- PHP error

**Fix:**
```bash
# Check permissions
chmod 644 public/css/custom-theme.css

# Check PHP errors
tail -f /var/log/apache2/error.log
# or
tail -f /var/log/php-fpm/error.log

# Test header.php loads
php -l templates/header.php
```

### Issue: System Name Not Changing

**Causes:**
- Not refreshing page
- Database not updated
- Function not working

**Fix:**
```bash
# Check database
mysql -u scubatricky_risk -p scubatricky_risk -e \
  "SELECT * FROM system_settings WHERE setting_key = 'system_name';"

# Should show your custom name

# Hard refresh browser
Ctrl+F5 or Cmd+Shift+R
```

## Best Practices

### Logo Design:
- ✅ Use vector format (SVG) for best quality
- ✅ Transparent background
- ✅ Simple, recognizable design
- ✅ Test on both dark and light backgrounds
- ✅ Keep text readable at small sizes

### Color Selection:
- ✅ Ensure sufficient contrast
- ✅ Test with color-blind simulation
- ✅ Maintain accessibility (WCAG AA)
- ✅ Keep brand consistency
- ✅ Test on actual content

### Testing:
- ✅ Test logo on all pages
- ✅ Test colors with all risk levels
- ✅ Check PDF reports
- ✅ Verify mobile display
- ✅ Test with multiple browsers

## Migration from Default

**Before:**
- Default orange theme
- ⚠️ icon logo
- "Risk Assessment" name

**After:**
- Your brand colors
- Your organization logo
- Your system name

**Steps:**
1. Prepare logo file (PNG/SVG, 200x50px)
2. Choose color scheme or use preset
3. Upload logo
4. Save colors
5. Test on all pages
6. Gather user feedback
7. Adjust if needed

## API Access (Optional)

### Get Current Settings:

```php
// Get system name
$stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = 'system_name'");
$stmt->execute();
$name = $stmt->fetchColumn();

// Get colors
$stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = 'color_scheme'");
$stmt->execute();
$colors = json_decode($stmt->fetchColumn(), true);
```

### Update Settings:

```php
// Update system name
$stmt = $pdo->prepare("
    INSERT INTO system_settings (setting_key, setting_value, updated_by)
    VALUES ('system_name', ?, ?)
    ON DUPLICATE KEY UPDATE setting_value = ?, updated_by = ?
");
$stmt->execute([$name, $userId, $name, $userId]);
```

## Backup & Restore

### Backup Customization:

```bash
# Backup logo files
tar -czf branding-backup.tar.gz public/uploads/branding/

# Backup custom CSS
cp public/css/custom-theme.css custom-theme-backup.css

# Backup database settings
mysqldump -u scubatricky_risk -p scubatricky_risk system_settings > settings-backup.sql
```

### Restore Customization:

```bash
# Restore logo
tar -xzf branding-backup.tar.gz -C public/uploads/

# Restore CSS
cp custom-theme-backup.css public/css/custom-theme.css

# Restore database
mysql -u scubatricky_risk -p scubatricky_risk < settings-backup.sql
```

## Security

### File Upload Security:
- ✅ File type validation (only images)
- ✅ File size limit (2MB)
- ✅ Filename sanitization
- ✅ Directory traversal prevention
- ✅ Admin-only access

### Database Security:
- ✅ Prepared statements
- ✅ Input validation
- ✅ XSS prevention
- ✅ SQL injection protection
- ✅ Audit trail (updated_by)

## Summary

**Features:**
- ✅ Custom system name
- ✅ Logo upload
- ✅ Color customization
- ✅ 6 preset schemes
- ✅ Live preview
- ✅ Easy reset

**Benefits:**
- Brand consistency
- Professional appearance
- Organization identity
- User recognition
- Customized experience

**Requirements:**
- Admin access
- system_settings table
- uploads/branding directory
- Write permissions

**Files:**
- system-customization.php (UI)
- migration_system_settings.sql (database)
- templates/header.php (updated)
- css/custom-theme.css (generated)

---

**Customize your Risk Assessment System to match your brand!** 🎨✨
